/*
  conserv.c -- CONSERV

    Copyright (C) 1999-2005 Naohisa Goto <ngoto@gen-info.osaka-u.ac.jp>
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <unistd.h>
#include <stddef.h>
#include <time.h>
#include "mfstlib3.h"
#include "my_malloc.h"
#include "ifilter.h"

/* read input filter */
#define MAKE_IFILTER_NAMELIST
#include "only_ifilter.h"
#include "xmask_ifilter.h"
#include "case_ifilter.h"

#define STAT 1
#ifndef STAT_VERBOSE
#define STAT_VERBOSE 0
#endif /* STAT_VERBOSE */
#define RECORD_PARENT 0
#define CALC_VARIETY 1
#define CALC_SD 0
#define SHOW_SEQ 1
#define CALCV_FAST 1
#define RECORD_TIME 1
#define DEBUG_MEM 0
#define USE_BITFIELD 0

#ifndef TYPEDEF_UCHAR
#define TYPEDEF_UCHAR
typedef unsigned char uchar;
#endif /* TYPEDEF_UCHAR */

#if USE_BITFIELD
#define STRNO_BITS 23
#define FLAG_BITS 1
#define STRNO_T unsigned
#define FLAG_T unsigned
#define STRNO_BITFIELD : STRNO_BITS
#define FLAG_BITFIELD : FLAG_BITS
#define STRNO_BIT_MAX ((1 << STRNO_BITS) - 1)
#define STRNO_MAX (STRNO_BIT_MAX - 2)
#define STRNO_NULL (STRNO_BIT_MAX - 1)
#define STRNO_DIVERSITY STRNO_BIT_MAX
#else /* USE_BITFIELD */
typedef int STRNO_T;
typedef unsigned char FLAG_T;
#define STRNO_BITFIELD
#define FLAG_BITFIELD
#define STRNO_MAX (INT_MAX - 2)
#define STRNO_NULL (INT_MAX - 1)
#define STRNO_DIVERSITY (INT_MAX)
#endif /* USE_BITFIELD */

typedef struct internode INTER;
typedef struct leafnode LEAF;

typedef union {
  INTER *inter;
  LEAF *leaf;
} NODEPTR;

typedef unsigned int NODEID;

/* node of suffix tree */
struct internode {
  /* common */
  int s; /* start position: base is 0 */
#if RECORD_PARENT
  NODEID parent; /* walk up the tree (direction of root) */
#endif
  NODEID neighbor;
  /* internode specific */
  int len; /* length */
  NODEID child;
  /* NODEID suffixlink; */
  /* STRNO_T strno; */
};

struct leafnode {
  /* common */
  int s; 
#if RECORD_PARENT
  NODEID parent;
#endif
  NODEID neighbor;
  /* leafnode specific */
};

#define NODE_IS_INTER 0

#define FLAG_RC_NO 0
#define FLAG_RC_YES 1
#define FLAG_RC_BOTH 2

#define BOOL_TRUE 1
#define BOOL_FALSE 0

/* macros */
#define m_main4_flagstr(flag_rc) ((flag_rc) ? "c" : " ")
#define m_main4_flagstr2(flag_rc) ((flag_rc) ? "c" : "1")
#define m_larger(x, y) ((x) > (y) ? (x) : (y))






extern int *g_strlen;

#define NODEID_NULL ((NODEID)0x7fffffff)

static int m_stb_node_getlen(NODEID id);
static int m_stb_node_getlen_inter(NODEID id);
static int m_stb_node_getlen_leaf(NODEID id);

static STRNO_T m_NODE_getmember_strno(NODEID id);
/* static void m_NODE_setmember_strno(NODEID id, STRNO_T val_strno); */
static NODEID m_NODE_getmember_suffixlink(NODEID id);
static void m_NODE_setmember_suffixlink(NODEID id, NODEID val_suffixlink);
static unsigned char m_NODE_getmember_leftchar(NODEID id);
static void m_NODE_setmember_leftchar(NODEID id, unsigned char val_lc);

static int m_stb_self_is(NODEID n);
static int m_NODE_getmember_s(NODEID id);
static void m_NODE_setmember_s(NODEID id, int val_s);
static void m_NODE_add_member_s(NODEID id, int val);
static NODEID m_NODE_getmember_neighbor(NODEID id);
static void m_NODE_setmember_neighbor(NODEID id, NODEID val_neighbor);
static int m_NODE_getmember_len(NODEID id);
static void m_NODE_setmember_len(NODEID id, int val_len);
static void m_NODE_sub_member_len(NODEID id, int val);
static NODEID m_NODE_getmember_child(NODEID id);
static void m_NODE_setmember_child(NODEID id, NODEID val_child);

#define m_stb_getwholestrlen(strno) (g_strlen[strno])
/* note: m_stb_getwholestrlen(strno) includes the length of last '\0' */

#if RECORD_PARENT
#define m_stb_add_child(sn, nn) \
 ((nn).leaf->neighbor = (sn)->child \
 , (sn)->child = (nn) \
 , (nn).leaf->parent = (sn)\
)
#else
#define m_stb_add_child(sn, nn) \
( m_NODE_setmember_neighbor((nn), m_NODE_getmember_child(sn)), \
  m_NODE_setmember_child((sn), (nn)) \
)
#endif

extern uchar **g_str;
#define m_stb_getwholestring(strn) \
 (g_str[strn])
#define m_stb_node_getstr(n) \
 (m_stb_getwholestring(m_NODE_getmember_strno(n)) + m_NODE_getmember_s(n))

/* *** check node struct *** */
int check_node_struct(void)
{
#if STAT
  fprintf(stderr, "sizeof(INTER, LEAF) = (%d, %d)\n",
	  sizeof(INTER), sizeof(LEAF));
#endif /* STAT */

  return 1;
} /* end of func */

/* global variables for option */
enum OutputMode {
  output_normal,
  output_mixed,
  output_old
};
int g_output_mode = output_normal;
int g_opt_output_append = 0;
int g_opt_output_overwrite = 0;
int g_opt_min_info = 1;
int g_opt_max_info; /* currently not used */
int g_opt_min_output = 100;
int g_opt_max_output; /* currently not used */
int g_opt_min_variety = 2;
int g_opt_max_variety; /* currently not used */
int g_opt_min_ncount = 1;
int g_opt_max_ncount; /* currently not used */
int g_opt_max_seqshowlen = 50;
#define G_OPT_MAX_SEQSHOWLEN_MAX INT_MAX
int g_opt_fastawidth = 70;
#define EXT_FST ".fst"
#define EXT_INF ".seqinfo"
#define EXT_POS ".position"
#define NAME_STDIN "(stdin)"
#define NAME_STDOUT "(stdout)"

#define OPT_MAGIC_ALL (-1)

#if STAT
size_t g_totalmem = 0;
int g_count_internode = 0;
int g_count_internode_pseudo = 0;
int g_count_leaf = 0;
int g_count_leaf_same = 0;

#if STAT_VERBOSE
int g_count_findchild = 0;
int g_count_findchild_call = 0;
#endif /* STAT_VERBOSE */

int g_maxdepth = 0;
int g_maxlen_internode = 0;
#endif /* STAT */

/* global for suffix tree */
extern int g_currentstrno;

/* global values */
#define CHARSETSIZE 256
#define END_OF_STR 0 /* also means node is leftdiversity */
#define LEFTCHAR_NULL (CHARSETSIZE - 1)

/* struct for main routine */
struct seq_source {
  struct seq_source *next;
  SEQUENCE *s;
  uchar *filename;
  int flag;
  int gid;
  int strno_p;
  int rest;
};

struct seq_list {
  struct seq_list *next;
  struct seq_source *source;
  struct ifilter_result *ifr;
  int part; /* base 1 */
};

#define GID_INIT 0

/* prototypes for main routine */
int main1_read_by_filename(int strno, struct seq_source **s_current,
 int flag_rc, int *gid_d, uchar *filename, int input_filetype);
FILE *main1_determin_outputfile_with_ext(const char *filename,
 const char *ext);

void main2_set_strlens(struct seq_list *list);

FILE *main2_determin_outputfile(const char *filename);
void main2_close_all_seq(struct seq_source *list);
void main2_destroy_seqlist(struct seq_list *list);
NODEID main2_buildsuffixtree(struct seq_list *list);
int main2_read(int strno, struct seq_source **s_current, 
 int flag_rc, int *gid_d,
 uchar *filename, FILE *fp_in);

void main2_output_result(NODEID root, struct seq_list *list, int ns,
 int groupmode, FILE *fp_out1, FILE *fp_out, FILE *fp_fst);

int main2_howmanyfiles(struct seq_source *list);
void main2_output_seqname(struct seq_source *list, int groupmode,
 FILE *fp_out);
void main2_output_seqinfo(struct seq_source *list, FILE *fp_out);
void main2_output_seqname_old(struct seq_source *list, int groupmode,
 FILE *fp_out);
void main2_output_seqinfo_old(struct seq_source *list, FILE *fp_out);

int main2_ifilter(struct seq_list **seqlist, struct seq_source *src,
 struct ifilter_activelist *alist);

int main3_compare_strings(char *str, char **cstr);

NODEID main3_build_first(uchar *str, int cstrlen);
NODEID main3_build_next(NODEID root, uchar *str, int cstrlen);
int main3_count_content(SEQUENCE *s0, int *t0);
void main3_print_content(struct seq_source *list, int *t0, int others,
 FILE *fp_out);
void main3_print_content_new(struct seq_source *list, int *t0, int others,
 FILE *fp_out);
int main3_check_ACGT(int *t0, int others, int *t0_prev, int others_prev);
struct seq_source *main3_read_seq(FILE *fp_in);
struct seq_source *main3_copy_seq(struct seq_source *src);
void main3_convert(SEQUENCE *s, int flag_rc);
void main3_printout(struct seq_source *list, FILE *fp_out);
void main3_printout_new(struct seq_source *list, FILE *fp_out);
void main3_output_result_inf(int maxlen, int minlen_stat,
 int *stat_c, int *stat_v, FILE *fp_out);

int main3_ifilter(struct seq_source *src, struct seq_list **list_current,
 struct ifilter_activelist *alist);

int *main3_make_counttable(struct seq_list *list, int ns, int *grps);
void main3_free_counttable(int *table);
int *main3_make_flagtable_strnotable(struct seq_list *list, int ns);
void main3_free_flagtable_strnotable(int *table);
int *main3_make_offsettable(struct seq_list *list, int ns);
void main3_free_offsettable(int *table);

/* prototypes for suffix tree */
NODEID stb_create_root(void);
NODEID stb_create_leaf(NODEID /*INTER*/ treenode, int st, int strn);
NODEID stb_create_internode(NODEID /*INTER*/ parent, NODEID treenode,
 int offset, int new_s);

void stb_add_child(NODEID /*INTER*/ source, NODEID newnode);
void stb_replace_child(NODEID /*INTER*/ source, NODEID old, NODEID new);
void stb_del_child(NODEID /*INTER*/ source, NODEID old);

NODEID stb_scan(NODEID sroot, int strno, int st, int len, 
 int *matchlen, int *offset, NODEID *parent);
NODEID stb_rescan(NODEID sroot, int strno, int st, int len, 
 int *matchlen, int *offset, NODEID *parent);
NODEID stb_find_child(NODEID /*INTER*/ p, uchar ch);

#ifdef MCCREIGHT
NODEID st_buildsuffixtree_mccreight(uchar *str, int len);
NODEID stb_mcc_scan(NODEID sroot, uchar *str, int st, int len, 
 int *matchlen, int *offset, NODEID *parent);
NODEID stb_mcc_rescan(NODEID sroot, uchar *str, int st, int len, 
 int *matchlen, int *offset, NODEID *parent);
NODEID stb_mcc_find_child(NODEID /*INTER*/ p, uchar *str, uchar ch);
#endif /* MCCREIGHT */

NODEID st_buildsuffixtree_ukkonen(uchar *str, int len);
int stb_ukk_singlephase(NODEID root, uchar *str, int edpos, int leafs,
 int strno, NODEID *r3node, int *r3pos);

NODEID st_generalizedsuffixtree_ukkonen(NODEID root, uchar *str, int len);
void stb_ukk_generalized_lastphase(NODEID root, NODEID r3node, int r3pos,
 int leafs, uchar *str, int strno, int len);

int stb_mark_leftdiversity(NODEID root, int *maxlen_internode);

void stb_output_identical_string(NODEID root, int maxdepth, int maxlen,
 int minlen_out, int minlen_stat, int min_variety, int min_ncount,
 int *stat_c, int *stat_v,
 int *counttable, int *flagtable, int *strnotable,
 int *offsettable, int *offsettable_c,
 int strs, int grps,
 int groupmode, enum OutputMode output_mode,
 FILE *fpo, FILE *fp_fst);

int stb_get_leafno(NODEID leaf);
NODEID stmem_new_leaf(void);
NODEID stmem_new_internode(void);

void stb_initialize_memadm_part1(int num_of_strings);
void stb_initialize_memadm_part2(int num_of_strings);
void stb_init_before_single_string(uchar *str, int size, int strno);
void stb_init_after_single_string(int strno);

void stmem_free_suffixlink(void);

/* ******************************************************************** */
void help_message(char *str)
{
#ifdef MCCREIGHT
  fprintf(stderr, "suffix tree algorithm : McCreight and Ukkonen\n");
#else /* MCCREIGHT */
  fprintf(stderr, "suffix tree algorithm : Ukkonen\n");
#endif /* MCCREIGHT */
  fprintf(stderr, "Usage: %s [option...] input.fst\n", str);
  fprintf(stderr, 
	  "-L nnn              "
	  "minimum length (default 100)\n"
	  "-v nnn              "
	  "minimum number of sequences or groups (default 2)\n"
	  "-n nnn              "
	  "minimum appearance count (default 1)\n"
	  "-i file.fst         "
	  "read sequence(s) from file.fst\n"
	  "-c file.fst         "
	  "read sequence(s) from file.fst, "
	  "using complementary strand only\n"
	  "-b file.fst         "
	  "read sequence(s) from file.fst, "
	  "using both strands\n"
	  "-t filter[=option]  "
	  "add input filter, or change filter settings\n"
	  "-o file             "
	  "specify output file prefix\n"
	  "-C | -B             "
	  "using {complemental | both} strands in all input sequence"
	  " (except files specified with -i or -c or -b options)\n"
	  "-g +                "
	  "group mode enabled, and change group\n"
	  "-g -                "
	  "group mode disabled, (1 sequence 1 group)\n"
	  "-g f                "
	  "group mode enabled, 1 file 1 group\n"
	  "\n"
	  "-x {mix|old}        "
	  "{mixed|old} output format (default: separate)\n"
	  "-x {fopen|popen}    "
	  "input files are {normal files(default)|popen}\n"
	  "-a | -F             "
	  "append | overwrite output file"
	  "(default: not append nor overwrite)\n"
	  "-O file             "
	  "specify output FASTA-format sequence file\n"
	  "-S nnn              "
	  "maximum length to show sequence in output file"
	  " (0 don\'t show, -1 show all) (default 50)\n"
	  "-l nnn              "
	  "minimum length of stat info output(default 1)\n" 
	  );

  fprintf(stderr, "\n");
  fprintf(stderr, 
	  "Input files: *.fst\n"
	  "Output files: file.seqinfo file.position file.fst (normal),\n"
	  "           or file (-x mix),\n"
	  "           or file file.fst (-x old)\n"
	  );
  fprintf(stderr, "\n");
  fprintf(stderr, "default loaded input filter: case=upper\n");

  return;
} /* end of func */

int main(int argc, char *argv[])
{
  int c, r;
  int tmpint0, tmpint1;
  char *str;
  FILE *fp_in;
  FILE *fp_out1 = stdout;
  FILE *fp_out2 = stdout;
  FILE *fp_fst = NULL;
  char *filename_out = NULL;
  char *filename_fst = NULL;
  int input_filetype = 0; /* 0: normal file 1: popen */

  int ns = 0;
  int ns_f = 0;
  int ns_p = 0;
  int ng_p = 0;
  int howmanyfiles;
  NODEID root;
  struct seq_list *seqlist;
  struct seq_source *s_source;
  struct seq_source s_top;
  struct seq_source *s_current = &s_top;
  int flag_rc = FLAG_RC_NO;
  int opt_allseqflag = FLAG_RC_NO;
  int gid_d[3] = { 1, 0, 0 };
  int groupmode = BOOL_FALSE;
  struct ifilter_activelist *alist = NULL;
  struct ifilter_namelist *ifnlist;
  enum{
    input_amino,
    input_nucleic
  } input_mode = input_nucleic;

  char *xoption_strings[] = {
    "mix",
    "old",
    "amino",
    "nucleic",
    "fopen",
    "popen",
    NULL
  };

#if RECORD_TIME
  time_t starttime, endtime;
  time_t starttime_tree, endtime_tree;
  long elapsed;
  long elapsed_tree;
#endif /* RECORD_TIME */

#if RECORD_TIME
  starttime = time(NULL);
#endif /* RECORD_TIME */

  /* ifilter name init */
  ifnlist = IFILTER_NAMELIST_LISTALL;

  s_top.s = NULL;
  s_top.next = NULL;
  s_top.gid = GID_INIT;
  s_top.strno_p = 0;

  if (check_node_struct() == 0) {
    fprintf(stderr, "node structure error.\n");
    exit(999);
  }

  fprintf(stderr, "Step 1: initialize/prepare\n");
  fprintf(stderr, "Substep 1: reading all sequences into memory...\n");
  alist = ifilter_registration("case=upper", ifnlist, alist);
  while (1) {
    /* opterr = 0; */
    c = getopt(argc, argv, "L:l:v:n:c:i:b:t:S:o:O:g:aFBChx:");
    if (c == EOF) break;

    switch (c) {
    case 'h':
      help_message(argv[0]);
      ifilter_print_help(ifnlist, stderr);
      exit(0);
      break;
    case 'l':
      r = sscanf(optarg, "%d..%d", &tmpint0, &tmpint1);
      switch (r) {
      case 2:
	if (tmpint1 <= 0) tmpint1 = 0;
	g_opt_max_info = tmpint1;
	/* fall into next case */
      case 1:
	if (tmpint0 <= 0) tmpint0 = 1;
	g_opt_min_info = tmpint0;
	break;
      default:
	fprintf(stderr, "Error in option -l ;  ignored.\n");
	break;
      }
      break;
    case 'L':
      r = sscanf(optarg, "%d..%d", &tmpint0, &tmpint1);
      switch (r) {
      case 2:
	if (tmpint1 <= 0) tmpint1 = 0;
	g_opt_max_output = tmpint1;
	/* fall into next case */
      case 1:
	if (tmpint0 <= 0) tmpint0 = 1;
	g_opt_min_output = tmpint0;
	break;
      default:
	fprintf(stderr, "Error in option -L ;  ignored.\n");
	break;
      }
      break;
    case 'v':
      r = sscanf(optarg, "%d..%d", &tmpint0, &tmpint1);
      switch (r) {
      case 2:
	if (tmpint1 <= 0) tmpint1 = 0;
	g_opt_max_variety = tmpint1;
	/* fall into next case */
      case 1:
	if (tmpint0 <= 0) tmpint0 = 1;
	g_opt_min_variety = tmpint0;
	break;
      default:
	if (strcmp(optarg, "all") == 0) {
	  g_opt_min_variety = OPT_MAGIC_ALL;
	  break; /* switch (r) */
	}
	fprintf(stderr, "Error in option -v ;  ignored.\n");
	break;
      }
      break;
    case 'n':
      r = sscanf(optarg, "%d..%d", &tmpint0, &tmpint1);
      switch (r) {
      case 2:
	if (tmpint1 <= 0) tmpint1 = 0;
	g_opt_max_ncount = tmpint1;
	/* fall into next case */
      case 1:
	if (tmpint0 <= 0) tmpint0 = 1;
	g_opt_min_ncount = tmpint0;
	break;
      default:
	fprintf(stderr, "Error in option -n ;  ignored.\n");
	break;
      }
      break;
    case 'c':
      input_mode = input_nucleic; /* input is nucleic */
      flag_rc = FLAG_RC_YES;
      /* fall into next case */
    case 'i':
      ns += main1_read_by_filename(ns, &s_current, flag_rc,
				   gid_d, optarg, input_filetype);
      flag_rc = FLAG_RC_NO;
      break;
    case 'b':
      input_mode = input_nucleic; /* input is nucleic */
      flag_rc = FLAG_RC_BOTH;
      ns += main1_read_by_filename(ns, &s_current, flag_rc,
				   gid_d, optarg, input_filetype);
      flag_rc = FLAG_RC_NO;
      break;
    case 'B':
      opt_allseqflag = FLAG_RC_BOTH;
      break;
    case 'C':
      opt_allseqflag = FLAG_RC_YES;
      break;
    case 't':
      if (optarg != NULL) {
	alist = ifilter_registration(optarg, ifnlist, alist);
      } else {
	fprintf(stderr, "Syntax error in option -t ;  ignored.\n");
      }
      break;
    case 'g':
      if (optarg != NULL) {
	switch (optarg[0]) {
	case '+':
	  groupmode = BOOL_TRUE;
	  gid_d[0] = 0;
	  gid_d[1] = 1;
	  gid_d[2] = 0;
	  break;
	case '-':
	  gid_d[0] = 1;
	  gid_d[1] = 0;
	  gid_d[2] = 0;
	  break;
	case 'f':
	  groupmode = BOOL_TRUE;
	  gid_d[0] = 0;
	  gid_d[1] = 0;
	  gid_d[2] = 1;
	  break;
	default:
	  fprintf(stderr, "Syntax error in option -g ;  ignored.\n");
	}
      } else {
	fprintf(stderr, "Syntax error in option -g ;  ignored.\n");
      }
      break;
    case 'S':
      r = sscanf(optarg, "%d", &tmpint0);
      if (r != 1 || tmpint0 < -1) {
	fprintf(stderr, "Error in option -S ;  ignored.\n");
      } else if (tmpint0 == -1) {
	g_opt_max_seqshowlen = G_OPT_MAX_SEQSHOWLEN_MAX;
      } else {
	g_opt_max_seqshowlen = tmpint0;
      }
      break;
    case 'F':
      g_opt_output_overwrite = 1;
      break;
    case 'a':
      g_opt_output_append = 1;
      break;
    case 'o':
      filename_out = optarg;
      break;
    case 'O':
      filename_fst = optarg;
      break;
    case 'x':
      r = main3_compare_strings(optarg, xoption_strings);
      switch (r) {
      case 0: /* mix */
	g_output_mode = output_mixed;
	break;
      case 1: /* old */
	g_output_mode = output_old;
	break;
      case 2: /* amino */
	input_mode = input_amino; /* non_nucleic(amino) */
	break;
      case 3: /* nucleic */
	input_mode = input_nucleic; /* nucleic */
	break;
      case 4: /* fopen */
	input_filetype = 0; /* normal file */
	break;
      case 5: /* popen */
	input_filetype = 1; /* popen */
	break;
      case -1:
      default:
	fprintf(stderr, "Unknown option -%c %s\n", c, optarg);
	break;
      } /* switch (r) */
      break;
    case '?':
      /* error occurs */
      break;
    default:
      fprintf (stderr, "?? getopt returned character code 0x%x(%c)\n", c, c);
    } /* switch (c) */
  } /* while (1) */

  while (optind < argc) {
    str = argv[optind++];
    ns += main1_read_by_filename(ns, &s_current, opt_allseqflag,
				 gid_d, str, input_filetype);
  } /* while (optind) */

  if (ns <= 0) {
    fprintf(stderr, "read sequences from stdin...\n");
    fp_in = stdin;
    ns += main2_read(ns, &s_current, opt_allseqflag, gid_d,
		     NAME_STDIN, fp_in);
  } /* if (ns == 0) */

  if (ns <= 0) {
    fprintf(stderr, "Error: not found sequence.\n");
    exit(1);
  }
  ns_p = s_current->strno_p;
  ng_p = s_current->gid;
  s_source = s_top.next;
  howmanyfiles = main2_howmanyfiles(s_source);
  fprintf(stderr, "\n%d files read\n", howmanyfiles);
  fprintf(stderr, "%d sequences (%d strands)\n", ns_p, ns);
  if (groupmode) fprintf(stderr, "%d groups\n", ng_p);
  fprintf(stderr, "\n");

  if (g_opt_min_variety == OPT_MAGIC_ALL) {
    g_opt_min_variety = ng_p;
  }

  fprintf(stderr, "Substep 2: prepare output file...\n");

 DETERMIN_OUTPUT:
  switch (g_output_mode) {
  case output_normal:
    if (filename_out == NULL) {
      g_output_mode = output_mixed;
      goto DETERMIN_OUTPUT;
    }
    fp_out1 = main1_determin_outputfile_with_ext(filename_out, EXT_INF);
    if (fp_out1 == NULL) {
      fprintf(stderr, "Error: cannot output; abort\n");
      exit(2);
    }
    fprintf(stderr, "Output file (sequence info): %s%s\n",
	    filename_out, EXT_INF);
    fp_out2 = main1_determin_outputfile_with_ext(filename_out, EXT_POS);
    if (fp_out2 == NULL) {
      fprintf(stderr, "Error: cannot output; abort\n");
      exit(2);
    }
    fprintf(stderr, "Output file (results): %s%s\n",
	    filename_out, EXT_POS);
    fp_fst = main1_determin_outputfile_with_ext(filename_out, EXT_FST);
    if (fp_fst != NULL) {
      fprintf(stderr, "Output FASTA file: %s%s\n", filename_out, EXT_FST);
    }
    break; /* switch */
  case output_mixed:
    fprintf(stderr, "mixed output\n");
    g_opt_max_seqshowlen = 0;
    if (filename_out != NULL) {
      fp_out1 = main2_determin_outputfile(filename_out);
      if (fp_out1 == NULL) {
	fprintf(stderr, "Error: cannot output; abort\n");
	exit(2);
      }
    } else {
      filename_out = NAME_STDOUT;
      fp_out1 = stdout;
    }
    fp_out2 = fp_out1;
    fprintf(stderr, "Output file (mix) : %s\n", filename_out);
    fp_fst = NULL;
    break; /* switch */
  case output_old:
    fprintf(stderr, "old style output\n");
    if (filename_out != NULL) {
      fp_out1 = main2_determin_outputfile(filename_out);
      if (fp_out1 == NULL) {
	fprintf(stderr, "Error: cannot output; abort\n");
	exit(2);
      }
    } else {
      filename_out = NAME_STDOUT;
      fp_out1 = stdout;
    }
    fp_out2 = fp_out1;
    fprintf(stderr, "Output file : %s\n", filename_out);
    if (filename_fst != NULL) {
      fp_fst = main2_determin_outputfile(filename_fst);
      if (fp_fst == NULL) {
	fprintf(stderr, "cannot output FASTA format file; abort\n");
	exit(2);
      }
      fprintf(stderr, "Output FASTA file: %s\n", filename_fst);
    } else if (filename_out != NULL && filename_out != NAME_STDOUT) {
      fp_fst = main1_determin_outputfile_with_ext(filename_out, EXT_FST);
      if (fp_fst == NULL) {
	fprintf(stderr, "cannot output FASTA format file; abort\n");
	exit(2);
      }
      fprintf(stderr, "Output FASTA file: %s%s\n", filename_out, EXT_FST);
    } else {
      fprintf(stderr, "Warning: Don\'t output FASTA format data\n");
    }
    break; /* switch */
  default:
    fprintf(stderr, "Internal error: switch(output_mode) %d\n", g_output_mode);
    abort();
  } /* switch (output_mode) */

  if (g_output_mode != output_old) {
    fprintf(fp_out1, "# input filter information\n");
    ifilter_print_activelist(alist, fp_out1, "% ");
    fprintf(fp_out1, "%% end\n");

    fprintf(fp_out1, "# general information\n");
    if (g_output_mode == output_mixed) {
      fprintf(fp_out1, "%% mixed_output\n");
    } else {
      fprintf(fp_out1, "%% separate_output\n");
    }
    switch (input_mode) {
    case input_nucleic:
      fprintf(fp_out1, "%% nucleic_mode\n");
      break;
    case input_amino:
    default:
      fprintf(fp_out1, "%% amino_mode\n");
      break;
    }
    if (groupmode) fprintf(fp_out1, "%% group_mode\n");
    else fprintf (fp_out1, "%% normal_mode\n");

    fprintf(fp_out1, "%% sequences=%d (%d)\n", ns_p, ns);
    if (groupmode) fprintf(fp_out1, "%% groups=%d\n", ng_p);
    else fprintf(fp_out1, "%%\n");
    fprintf(fp_out1, "%% files=%d\n", howmanyfiles);
    fprintf(fp_out1, "%% end\n");
    fprintf(fp_out1, "# input sequences\n");
    main2_output_seqname(s_source, groupmode, fp_out1);
    fprintf(fp_out1, "%% end\n");

    if (input_mode == input_nucleic) {
      fprintf(fp_out1, "# input sequence statistics\n");
      main2_output_seqinfo(s_source, fp_out1);
      fprintf(fp_out1, "%% end\n");
    } else {
      fprintf(fp_out1, "#\n");
      fprintf(fp_out1, "%% end\n");
    }
  } else { /* old output */
    fprintf(fp_out1, "# old(compatible) output\n");
    ifilter_print_activelist(alist, fp_out1, "# ");
    fprintf(fp_out1, "# input sequences\n");
    if (groupmode) fprintf(fp_out1, "# group mode\n");
    else fprintf (fp_out1, "# normal mode (1 sequence 1 group)\n");

    fprintf(fp_out1, "%% total_ns=%d nl=%d\n", ns_p, ns);
    if (groupmode) fprintf(fp_out1, "%% total_groups=%d\n", ng_p);
    fprintf(fp_out1, "%% total_files=%d\n", howmanyfiles);
    main2_output_seqname_old(s_source, groupmode, fp_out1);

    fprintf(fp_out1, "# ATGC content\n");
    main2_output_seqinfo_old(s_source, fp_out1);
  }

  seq_garbage_collection();

  fprintf(stderr, "Substep 3: input filters...\n");
  ns_f = main2_ifilter(&seqlist, s_source, alist);

  stb_initialize_memadm_part1(ns_f);
  main2_set_strlens(seqlist);
  stb_initialize_memadm_part2(ns_f);

  fflush(fp_out1); /* flush seqinfo file */

#if RECORD_TIME
  starttime_tree = time(NULL);
#endif /* RECORD_TIME */
  fprintf(stderr, "Step 2: building generalized suffix tree...\n");
  root = main2_buildsuffixtree(seqlist);
#if RECORD_TIME
  endtime_tree = time(NULL);
#endif /* RECORD_TIME */

  fprintf(stderr, "Freeing memory of suffix link...\n");
  stmem_free_suffixlink();
#if STAT
  fprintf(stderr, "total memory(including garbage): %lu\n", 
	  (unsigned long)g_totalmem);
#endif

  fprintf(stderr, "Step 3: output results...\n");
  main2_output_result(root, seqlist, ns_f, groupmode,
		      fp_out1, fp_out2, fp_fst);

  ifilter_close_all(alist);
  main2_destroy_seqlist(seqlist);
  main2_close_all_seq(s_source);
  if (fp_out1 != NULL && fp_out1 != stdout && fp_out1 != stderr) {
    fclose(fp_out1);
  }
  if (fp_out2 != fp_out1 &&
      fp_out2 != NULL && fp_out2 != stdout && fp_out2 != stderr) {
    fclose(fp_out2);
  }
  if (fp_fst != NULL && fp_fst != stdout && fp_fst != stderr) {
    fclose(fp_fst);
  }

   fprintf(stderr, "done.\n");
#if RECORD_TIME
  endtime = time(NULL);
  elapsed = (long)difftime(endtime, starttime);
  elapsed_tree = (long)difftime(endtime_tree, starttime_tree);
  fprintf(stderr, "time elapsed: %ld sec ( = %ld min %ld sec )\n",
	  elapsed,
	  elapsed / 60, elapsed % 60);
  fprintf(stderr, "time elapsed for tree: %ld sec ( = %ld min %ld sec )\n",
	  elapsed_tree,
	  elapsed_tree / 60, elapsed_tree % 60);
#endif /* RECORD_TIME */

  exit(0);
} /* end of main() */

/* ******************************************************************** */
int main3_compare_strings(char *str, char **cstr)
{
  char *s;
  int i;

  for (i = 0; ;i++) {
    s = *cstr++;
    if (s == NULL) {
      i = -1;
      break; /* for */
    }
    if (strcmp(s, str) == 0) break; /* for */
  } /* for */
  
  return i;
} /* end of func */

/* ******************************************************************** */
int main1_read_by_filename(int strno, struct seq_source **s_current,
 int flag_rc, int *gid_d, uchar *filename, int input_filetype)
{
  FILE *fp_in;
  int c = 0;

  if (filename != NULL) {
    switch (input_filetype) {
    case 1:
      fp_in = popen(filename, "r");
      break;
    case 0:
    default:
      fp_in = fopen(filename, "r");
      break;
    } /* switch */

    if (fp_in == NULL) {
      fprintf(stderr, "can\'t open file \"%s\"; ignored\n", filename);
    } else {
      fprintf(stderr, "reading %s\n", filename);
      c = main2_read(strno, s_current, flag_rc, gid_d, 
		     filename, fp_in);

      switch (input_filetype) {
      case 1:
	pclose(fp_in);
	break;
      case 0:
      default:
	fclose(fp_in);
	break;
      } /* switch */
    } /* if (fp_in...) */
  } /* if (str...) */
  return c;
} /* end of func */

FILE *main1_determin_outputfile_with_ext(const char *filename, const char *ext)
{
  char *fn_new;
  size_t n1, n2;
  FILE *fp;

  n1 = strlen(filename) + 1;
  n2 = strlen(ext) + 1;
  fn_new = my_malloc(n1 + n2);
  strncpy(fn_new, filename, n1);
  strncat(fn_new, ext, n2);

  fp = main2_determin_outputfile(fn_new);
  my_free(fn_new);
  return fp;
} /* end of func */

FILE *main2_determin_outputfile(const char *filename)
{
  FILE *fp = NULL;

  if (filename == NULL) {
    fprintf(stderr, "Output filename == NULL ??\n");
    return NULL;
  }

  if (g_opt_output_append) {
    fp = fopen(filename, "a");
  } else if (g_opt_output_overwrite) {
    fp = fopen(filename, "w");
  } else {
    fp = fopen(filename, "r");
    if (fp != NULL) {
      fprintf(stderr, "output file %s exists\n", filename);
      fclose(fp);
      return NULL;
    }
    fp = fopen(filename, "a");
  }
  if (fp == NULL) {
    fprintf(stderr, "can\'t open file %s\n", filename);
    return NULL;
  }
  
  return fp;
} /* end of func */

int main2_howmanyfiles(struct seq_source *list)
{
  uchar *fn = NULL;
  int nf = 0;

  while (list != NULL) {
    if (fn != list->filename) {
      nf += 1;
      fn = list->filename;
    }
    list = list->next;
  } /* while (list != NULL) */

  return nf;
} /* end of func */

void main2_output_seqname_old(struct seq_source *list, int groupmode,
 FILE *fp_out)
{
  struct seq_source *list2;
  int gid;
  uchar *fn = NULL;
  int ns = 0;
  int ns_p = 0;
  int ns_f = 0;
  int ns_f_p = 0;

  while (list != NULL) {
    if (groupmode && ns <= 0) {
      list2 = list;
      gid = list2->gid;
      ns = 0;
      ns_p = 0;
      while (gid == list2->gid) {
	ns += 1;
	if (list2->flag != FLAG_RC_BOTH) ns_p += 1;
	list2 = list2->next;
	if (list2 == NULL) break;
      }
      fprintf(fp_out, "%% g=%d ns=%d nl=%d\n",
	      list->gid, ns_p, ns);
    }
    if (ns_f <= 0) {
      list2 = list;
      fn = list2->filename;
      ns_f = 0;
      ns_f_p = 0;
      while (fn == list2->filename) {
	ns_f += 1;
	if (list2->flag != FLAG_RC_BOTH) ns_f_p += 1;
	list2 = list2->next;
	if (list2 == NULL) break;
      }
      fprintf(fp_out, "%% ns=%d nl=%d file=%s\n",
	      ns_f_p, ns_f, list->filename);
    }
    if (list->flag == FLAG_RC_BOTH) {
      fprintf(fp_out, "%d%s\n", list->strno_p, m_main4_flagstr(FLAG_RC_BOTH));
    } else {
      main3_printout(list, fp_out);
    }
    ns -= 1;
    ns_f -= 1;
    list = list->next;
  } /* while (list != NULL) */
  return;
} /* end of func */

void main2_output_seqinfo_old(struct seq_source *list, FILE *fp_out)
{
  SEQUENCE *s;
  int strno;
  int others;
  int flag_rc;
  int t[CHARSETSIZE];

  for (strno = 0; list != NULL; strno++) {
    s = list->s;
    flag_rc = list->flag;
    others = main3_count_content(s, t);
    if (others > 0) {
      fprintf(stderr, "Note: %d non-ATGC characters in %d%s\n",
	      others, list->strno_p,
	      m_main4_flagstr(list->flag));
    }
    main3_print_content(list, t, others, fp_out);
    list = list->next;
  } /* for */

  return;
} /* end of func */

void main2_output_seqname(struct seq_source *list, int groupmode,
 FILE *fp_out)
{
  struct seq_source *list2;
  int gid;
  uchar *fn = NULL;
  int ns = 0;
  int ns_p = 0;
  int ns_f = 0;
  int ns_f_p = 0;

  while (list != NULL) {
    if (groupmode && ns <= 0) {
      list2 = list;
      gid = list2->gid;
      ns = 0;
      ns_p = 0;
      while (gid == list2->gid) {
	ns += 1;
	if (list2->flag != FLAG_RC_BOTH) ns_p += 1;
	list2 = list2->next;
	if (list2 == NULL) break;
      }
      fprintf(fp_out, "%% g=%d ns=%d (%d)\n", list->gid, ns_p, ns);
    }
    if (ns_f <= 0) {
      list2 = list;
      fn = list2->filename;
      ns_f = 0;
      ns_f_p = 0;
      while (fn == list2->filename) {
	ns_f += 1;
	if (list2->flag != FLAG_RC_BOTH) ns_f_p += 1;
	list2 = list2->next;
	if (list2 == NULL) break;
      }
      fprintf(fp_out, "%% ns=%d (%d) file=%s\n",
	      ns_f_p, ns_f, list->filename);
    }
    main3_printout_new(list, fp_out);
    ns -= 1;
    ns_f -= 1;
    list = list->next;
  } /* while (list != NULL) */
  return;
} /* end of func */

void main2_output_seqinfo(struct seq_source *list, FILE *fp_out)
{
  SEQUENCE *s;
  int others, others_prev;
  int t1[CHARSETSIZE], t2[CHARSETSIZE];
  int *t, *t_prev, *t_tmp;
  int r;

  t = t1;
  t_prev = t2;
  others_prev = 0;

  while (list != NULL) {
    s = list->s;
    others = main3_count_content(s, t);
    if (others > 0) {
      fprintf(stderr, "Note: %d non-ACGT characters in %d%s\n",
	      others, list->strno_p,
	      m_main4_flagstr(list->flag));
    }
    main3_print_content_new(list, t, others, fp_out);
    if (list->flag == FLAG_RC_BOTH) {
      r = main3_check_ACGT(t, others, t_prev, others_prev);
      if (r != 0) {
	fprintf(stderr, "Warning: ACGT contents do not match"
		" between %d%s and %d%s\n",
		list->strno_p, m_main4_flagstr(FLAG_RC_NO),
		list->strno_p, m_main4_flagstr(list->flag));
	fprintf(fp_out, "%% Warning: ACGT contents not match\n");
      }
    }
    list = list->next;
    t_tmp = t;
    t = t_prev;
    t_prev = t_tmp;
    others_prev = others;
  } /* while */

  return;
} /* end of func */

void main2_close_all_seq(struct seq_source *list)
{
  struct seq_source *list_next;

  while (list != NULL) {
    if (list->s != NULL) seq_close(list->s);
    list_next = list->next;
    my_free(list);
    list = list_next;
  } /* while */

  return;
} /* end of func */

void main2_destroy_seqlist(struct seq_list *list)
{
  struct seq_list *list_next;

  while (list != NULL) {
    if (list->ifr != NULL) ifilter_result_free(list->ifr);
    list_next = list->next;
    my_free(list);
    list = list_next;
  } /* while */

  return;
} /* end of func */

void main2_set_strlens(struct seq_list *list)
{
  int i = 0;

  while (list != NULL) {
    g_strlen[i++] = list->ifr->len + 1; /* don't forget last '\0' */

    list = list->next;
  } /* while */
  /* fprintf(stderr, "%d\n", i); */ /* for DEBUG */
  return;
} /* end of func */

NODEID main2_buildsuffixtree(struct seq_list *list)
{
  NODEID root;

  fprintf(stderr, "seq %d%s part %d\n", list->source->strno_p,
	  m_main4_flagstr(list->source->flag),
	  list->part);
  root = main3_build_first(list->ifr->str, list->ifr->len);

#if STAT
#if STAT_VERBOSE
  fprintf(stderr, "find_child called %d times; %d loop\n",
	  g_count_findchild_call, g_count_findchild);
#endif /* STAT_VERBOSE */
  fprintf(stderr, "total internode = %d (%d true) leaf = %d (%d used)\n",
	  g_count_internode, g_count_internode - g_count_internode_pseudo,
	  g_count_leaf - g_count_leaf_same, g_count_leaf);
  fprintf(stderr, "total memory(including garbage): %lu\n",
	  (unsigned long)g_totalmem);
#endif /* STAT */

  while ((list = list->next) != NULL) {
    fprintf(stderr, "seq %d%s part %d\n",
	    list->source->strno_p,
	    m_main4_flagstr(list->source->flag),
	    list->part);
    root = main3_build_next(root, list->ifr->str, list->ifr->len);
#if STAT
#if STAT_VERBOSE
    fprintf(stderr, "find_child called %d times; %d loop\n",
	    g_count_findchild_call, g_count_findchild);
#endif /* STAT_VERBOSE */
    fprintf(stderr, "total internode = %d (%d true) leaf = %d (%d used)\n",
	    g_count_internode, g_count_internode - g_count_internode_pseudo,
	    g_count_leaf - g_count_leaf_same, g_count_leaf);
    fprintf(stderr, "total memory(including garbage): %lu\n",
	    (unsigned long)g_totalmem);
#endif /* STAT */
  } /* while */

  return root;
} /* end of func */

int main2_read(int strno, struct seq_source **s_current,
 int flag_rc, int *gid_d, uchar *filename, FILE *fp_in)
{
  int count;
  struct seq_source *p;
  int gid;
  int d;
  int strno_p;

  strno_p = (*s_current)->strno_p + 1;
  gid = (*s_current)->gid + gid_d[0] + gid_d[1] + gid_d[2];
  d = gid_d[0];

  if (flag_rc == FLAG_RC_BOTH) for (count = 0; ; count++, strno++,
				    strno_p++, gid += d) {
    p = main3_read_seq(fp_in);
    if (p == NULL) break;
    (*s_current)->next = p;
    *s_current = p;
    p->flag = FLAG_RC_NO;
    p->gid = gid;
    p->strno_p = strno_p;
    p->rest = 1;
    p->filename = filename;
    main3_printout(p, stderr);
    /* main3_convert(p->s, p->flag); */
    strno++, count++;
    p = main3_copy_seq(p);
    if (p == NULL) {
      fprintf(stderr, "cannot alloc memory.\n");
      exit(2);
    }
    (*s_current)->next = p;
    *s_current = p;
    p->flag = FLAG_RC_BOTH;
    p->gid = gid;
    p->strno_p = strno_p;
    p->rest = 0;
    p->filename = filename;
    /* main3_printout(p, stderr); */
    fprintf(stderr, "%d%s\n", strno_p, m_main4_flagstr(FLAG_RC_BOTH));
    main3_convert(p->s, p->flag);
  } else for (count = 0; ; count++, strno++, strno_p++, gid += d) {
    p = main3_read_seq(fp_in);
    if (p == NULL) break;
    (*s_current)->next = p;
    *s_current = p;
    p->flag = flag_rc;
    p->gid = gid;
    p->strno_p = strno_p;
    p->rest = 0;
    p->filename = filename;
    main3_printout(p, stderr);
    main3_convert(p->s, flag_rc);
  } 

  if (count > 0) gid_d[1] = 0;
  return count;
} /* end of func */

void main2_output_result(NODEID root, struct seq_list *list, int ns,
 int groupmode, FILE *fp_out1, FILE *fp_out, FILE *fp_fst)
{
  int maxdepth;
  int maxlen;
  int *stat_c, *stat_v;
  int *counttable;
  int *flagtable;
  int *strnotable;
  int *offsettable, *offsettable_c;
  int grps;

  fprintf(stderr, "Substep 1: trace all nodes and set leftdiversity\n");
  maxdepth = stb_mark_leftdiversity(root, &maxlen);

#if STAT
  g_maxdepth = maxdepth;
  fprintf(stderr, "max depth = %d\n", g_maxdepth);
  g_maxlen_internode = maxlen;
  fprintf(stderr, "max len from root to internal node = %d\n"
	  , g_maxlen_internode);
#endif /* STAT */

  counttable = main3_make_counttable(list, ns, &grps);
  flagtable = main3_make_flagtable_strnotable(list, ns);
  strnotable = flagtable + ns;
  offsettable = main3_make_offsettable(list, ns);
  offsettable_c = offsettable + ns;

  /* output results */
  fprintf(stderr, "Substep 2: output...\n");
  stat_c = my_calloc(maxlen + 1, sizeof(int));
  stat_v = my_calloc(maxlen + 1, sizeof(int));

  fprintf(fp_out, "# location data ( n>=%d L>=%d v>=%d )\n",
	  g_opt_min_ncount, g_opt_min_output, g_opt_min_variety);
  stb_output_identical_string(root, maxdepth, maxlen,
			      g_opt_min_output, g_opt_min_info,
			      g_opt_min_variety, g_opt_min_ncount,
			      stat_c, stat_v,
			      counttable, flagtable, strnotable,
			      offsettable, offsettable_c,
			      ns, grps, groupmode,
			      g_output_mode,
			      fp_out, fp_fst);
  if (g_output_mode != output_old) fprintf(fp_out, "%% end\n");

  fprintf(fp_out, "# len sort count ( n>=%d L>=%d v>=%d )\n",
	  g_opt_min_ncount, g_opt_min_info, g_opt_min_variety);
  main3_output_result_inf(maxlen, g_opt_min_info, stat_c, stat_v, fp_out);
  if (g_output_mode != output_old) fprintf(fp_out, "%% end\n");

  main3_free_offsettable(offsettable);
  main3_free_flagtable_strnotable(flagtable);
  main3_free_counttable(counttable);
  my_free(stat_v);
  my_free(stat_c);
  return;
} /* end of func */

/* ******************************************************************** */
int *main3_make_offsettable(struct seq_list *list, int ns)
{
  int i;
  int *table1;
  int *table2;

  table1 = my_malloc(2 * ns * sizeof(int));
  table2 = table1 + ns;

  for (i = 0; list != NULL; i++) {
    table1[i] = list->ifr->offset;
    table2[i] = list->ifr->offset_c;
    list = list->next;
  } /* for */

  return table1;
} /* end of func */

void main3_free_offsettable(int *table)
{
  my_free(table);
} /* end of func */

int *main3_make_flagtable_strnotable(struct seq_list *list, int ns)
{
  int i;
  int *table1;
  int *table2;

  table1 = my_malloc(2 * ns * sizeof(int));
  table2 = table1 + ns;

  for (i = 0; list != NULL; i++) {
    table1[i] = list->source->flag;
    table2[i] = list->source->strno_p;
    list = list->next;
  } /* for */

  return table1;
} /* end of func */

void main3_free_flagtable_strnotable(int *table)
{
  my_free(table);
} /* end of func */

int *main3_make_counttable(struct seq_list *list, int ns, int *grps)
{
  int i;
  int gid, gid2;
  int n;
  int *table = my_malloc(ns * sizeof(int));

  gid = GID_INIT;
  for (i = 0, n = -1; list != NULL; i++) {
    if ((gid2 =list->source->gid) != gid) {
      gid = gid2;
      n++;
    }
    table[i] = n;
    list = list->next;
  } /* for */

  *grps = n + 1;
  return table;
} /* end of func */

void main3_free_counttable(int *table)
{
  my_free(table);
} /* end of func */

void main3_output_result_inf(int maxlen, int minlen_stat,
 int *stat_c, int *stat_v, FILE *fp_out)
{
  int i;

  while (stat_v[maxlen] <= 0) maxlen--;
  
  for (i = minlen_stat; i <= maxlen; i++) {
    if (stat_v[i] > 0) {
      fprintf(fp_out, "%d\t%d\t%d\n", i, stat_v[i], stat_c[i]);
    }
  }
  return;
} /* end of func */

int main3_count_content(SEQUENCE *s0, int *t0)
{
  int i, others = 0;
  uchar countstring[CHARSETSIZE + 1];

  for (i = 0; i < CHARSETSIZE; i++) countstring[i] = i;
  countstring[CHARSETSIZE] = '\0';

  seq_count_content(s0, t0, countstring);
  for (i = 0; i < CHARSETSIZE; i++) others += t0[i];
  others -= (t0['A'] + t0['C'] + t0['G'] + t0['T']);
  others -= (t0['a'] + t0['c'] + t0['g'] + t0['t']);

  return others;
} /* end of func */

void main3_print_content(struct seq_source *list, int *t0, int others,
 FILE *fp_out)
{
  fprintf(fp_out, "%d%s total=%d A=%d C=%d G=%d T=%d other=%d\n",
	  list->strno_p, m_main4_flagstr(list->flag),
	  seq_getsize(list->s),
	  t0['A'] + t0['a'], 
	  t0['C'] + t0['c'],
	  t0['G'] + t0['g'],
	  t0['T'] + t0['t'],
	  others);
  return;
} /* end of func */

int main3_check_ACGT(int *t0, int others, int *t0_prev, int others_prev)
{
  int cA, cC, cG, cT;
  int pA, pC, pG, pT;

  cA = t0['A'] + t0['a']; 
  cC = t0['C'] + t0['c'];
  cG = t0['G'] + t0['g'];
  cT = t0['T'] + t0['t'];
  pA = t0_prev['A'] + t0_prev['a']; 
  pC = t0_prev['C'] + t0_prev['c'];
  pG = t0_prev['G'] + t0_prev['g'];
  pT = t0_prev['T'] + t0_prev['t'];
  if (pA != cT || pC != cG || pG != cC || pT != cA || others != others_prev)
    return -1;
  else
    return 0;
} /* end of func */

void main3_print_content_new(struct seq_source *list, int *t0, int others,
 FILE *fp_out)
{
  int cA, cC, cG, cT;

  cA = t0['A'] + t0['a']; 
  cC = t0['C'] + t0['c'];
  cG = t0['G'] + t0['g'];
  cT = t0['T'] + t0['t'];

  if (list->flag == FLAG_RC_BOTH) fprintf(fp_out, "%% ");
  fprintf(fp_out, "%d%s total=%d A=%d C=%d G=%d T=%d other=%d\n",
	  list->strno_p, m_main4_flagstr(list->flag),
	  seq_getsize(list->s), cA, cC, cG, cT,
	  others);

  return;
} /* end of func */

struct seq_source *main3_read_seq(FILE *fp_in)
{
  SEQUENCE *s;
  struct seq_source *p;
  int r;

  s = seq_open();
  r = get_fasta(s, fp_in);
  if (r != 0) {
    seq_close(s);
    return NULL;
  }

  p = my_calloc(1, sizeof(struct seq_source));
  p->s = s;
  p->next = NULL;

  return p;
} /* end of func */

struct seq_source *main3_copy_seq(struct seq_source *src)
{
  SEQUENCE *s;
  struct seq_source *p;
  int r;

  s = seq_open();
  r = seq_duplicate(src->s, s);
  if (r != 0) {
    seq_close(s);
    return NULL;
  }

  p = my_malloc(sizeof(struct seq_source));
  p->s = s;
  p->next = NULL;
  p->flag = 0;

  return p;
} /* end of func */

void main3_convert(SEQUENCE *s, int flag_rc) 
{
  if (flag_rc) {
    seq_convertall_reverse(s, get_convert_table(CONVERT_COMPLEMENT_DNA));
  }
  return;
} /* end of func */

void main3_printout(struct seq_source *list, FILE *fp_out)
{
  fprintf(fp_out, "%d%s %s\n", list->strno_p,
	  m_main4_flagstr(list->flag),
	  seq_getname(list->s));
  return;
} /* end of func */

void main3_printout_new(struct seq_source *list, FILE *fp_out)
{
  if (list->flag == FLAG_RC_BOTH) return;

  fprintf(fp_out, "%d %s", list->strno_p, m_main4_flagstr2(list->flag));
  if (list->rest > 0) fprintf(fp_out, "+%s", m_main4_flagstr2(FLAG_RC_YES));
  fprintf(fp_out, " L=%d //%d\n",
	  seq_getsize(list->s), seq_getnamelen(list->s));
  fprintf(fp_out, "%s\n", seq_getname(list->s));
  return;
} /* end of func */

/* ***************************************************************** */
int main2_ifilter(struct seq_list **seqlist, struct seq_source *src,
 struct ifilter_activelist *alist)
{
  int ns_f = 0;
  struct seq_list top;
  struct seq_list *current = &top;

  top.next = NULL;

  while (src != NULL) {
    ns_f += main3_ifilter(src, &current, alist);
    src = src->next;
  } /* while */

  *seqlist = top.next;
  return ns_f;
} /* end of func */

int main3_ifilter(struct seq_source *src, struct seq_list **list_current,
 struct ifilter_activelist *alist)
{
  struct seq_list *p;
  struct ifilter_result *ifr, *ifr_next;
  int rcount;

  ifr = ifilter_main(seq_getseq(src->s), seq_getsize(src->s), alist);
  rcount = 0;

  while (ifr != NULL) {
    rcount += 1;
    p = my_malloc(sizeof(struct seq_list));
    p->part = rcount; /* part No. starts from 1 */
    p->source = src;
    p->ifr = ifr;
    ifr_next = ifr->next;
    ifr->next = NULL;
    ifr->offset += 1; /* for output position adjustment */
    ifr->offset_c += 1; /* for output position adjustment */
    
    p->next = NULL;
    (*list_current)->next = p;
    *list_current = p;
    ifr = ifr_next;
  } /* while */

  return rcount;
} /* end of func */

/* ***************************************************************** */

/* ***************************************************************** */
NODEID main3_build_first(uchar *str, int cstrlen)
{
  NODEID root;

  cstrlen += 1; /* don't forget last '\0' */
  root = st_buildsuffixtree_ukkonen(str, cstrlen);
  return root;
} /* end of func */

NODEID main3_build_next(NODEID root, uchar *str, int cstrlen)
{
  cstrlen += 1; /* don't forget last '\0' */
  root = st_generalizedsuffixtree_ukkonen(root, str, cstrlen);
  return root;
} /* end of func */

/* ***************************************************************** */

/* ******************************************************************** */
NODEID stb_create_root(void)
{
  NODEID newnode;

  newnode = stmem_new_internode();
  return newnode;
} /* end of func */

NODEID stb_create_leaf(NODEID /*INTER*/ treenode, int st, int strn)
{
  NODEID newnode;

  newnode = stmem_new_leaf();
  /* m_NODE_setmember_strno(newnode, strn); */
  m_NODE_setmember_s(newnode, st);
  m_stb_add_child(treenode, newnode);

  return newnode;
} /* end of func */

#if 0
NODEID stb_create_internode(NODEID /*INTER*/ parent, NODEID treenode,
 int offset)
/*
image:
   (parnt)----(node)    --> (parent)------------(new)-----------(node)
          s   end                   s  s+offset-1    s+offset  end
*/
{
  NODEID newnode;

  newnode = stmem_new_internode();

  m_NODE_setmember_s(newnode, m_NODE_getmember_s(treenode));
  m_NODE_setmember_len(newnode, offset);
  m_NODE_setmember_strno(newnode, m_NODE_getmember_strno(treenode));

  m_NODE_add_member_s(treenode, offset);
  if (m_stb_self_is(treenode) == NODE_IS_INTER) {
    m_NODE_sub_member_len(treenode, offset);
  }

  stb_replace_child(parent, treenode, newnode);
  m_stb_add_child(newnode, treenode);

  return newnode;
} /* end of func */
#endif /* 0 */

NODEID stb_create_internode(NODEID /*INTER*/ parent, NODEID treenode,
 int offset, int new_s)
/*
image:
   (parnt)----(node)    --> (parent)-------------------(new)----------(node)
          s   end                   new_s new_s+offset-1   s+offset  end
*/
{
  NODEID newnode;

  newnode = stmem_new_internode();

  m_NODE_setmember_s(newnode, new_s);
  m_NODE_setmember_len(newnode, offset);

  m_NODE_add_member_s(treenode, offset);
  if (m_stb_self_is(treenode) == NODE_IS_INTER) {
    m_NODE_sub_member_len(treenode, offset);
  }

  stb_replace_child(parent, treenode, newnode);
  m_stb_add_child(newnode, treenode);

  return newnode;
} /* end of func */


void stb_add_child(NODEID /*INTER*/ source, NODEID newnode)
#if 1
{
  m_stb_add_child(source, newnode);
} /* end of func */
#endif
#if 0
{
  NODEID n = m_NODE_getmember_child(source);

  m_NODE_setmember_child(source, newnode);
  m_NODE_setmember_neighbor(newnode, n);

#if RECORD_PARENT
  m_NODE_setmember_parent(newnode, source);
#endif
  return;
} /* end of func */
#endif

void stb_replace_child(NODEID /*INTER*/ source, NODEID old, NODEID new)
#if 0
{
  stb_del_child(source, old);
  stb_add_child(source, new);
 return;
} /* end of func */
#endif
#if 1
{
  NODEID c, n, nn;

  m_stb_add_child(source, new);
  c = m_NODE_getmember_child(source);
  n = m_NODE_getmember_neighbor(c);

  while (n != NODEID_NULL) {
    nn = m_NODE_getmember_neighbor(n);
    if (n == old) {
      m_NODE_setmember_neighbor(c, nn);
      return;
    }
    c = n;
    n = nn;
  } /* while */

  abort();
  return;
} /* end of func */
#endif

void stb_del_child(NODEID /*INTER*/ source, NODEID old)
{
  NODEID c, n, nn;

  c = m_NODE_getmember_child(source);

#if 0 /* error check */
  if (c == NODEID_NULL) {
    abort();
  }
#endif /* 0 */

  n = m_NODE_getmember_neighbor(c);
  if (c == old) {
    m_NODE_setmember_child(source, n);
    return;
  }

  while (n != NODEID_NULL) {
    nn = m_NODE_getmember_neighbor(n);
    if (n == old) {
      m_NODE_setmember_neighbor(c, nn);
      return;
    }
    c = n;
    n = nn;
  } /* while */

  abort();
  return;
} /* end of func */

/* ******************************************************************** */
#ifdef MCCREIGHT
NODEID stb_mcc_find_child(NODEID p, uchar *str, uchar ch)
{
  NODEID c;
#if STAT_VERBOSE
  g_count_findchild_call += 1;
  g_count_findchild += 1;
#endif /* STAT_VERBOSE */

  c = m_NODE_getmember_child(p);
  while (c != NODEID_NULL) {
    if (str[m_NODE_getmember_s(c)] == ch) return c;
    c = m_NODE_getmember_neighbor(c);
#if STAT_VERBOSE
    g_count_findchild += 1;
#endif /* STAT_VERBOSE */
  }

  c = NODEID_NULL;
  return c;
} /* end of func */

NODEID st_buildsuffixtree_mccreight(uchar *str, int len)
/* len must contain length of last '\0' */
/* len must be >0 */
{
  const int strn = 0;
  NODEID root;
  NODEID parent;
  int i;
  int offset, substepb_offset, matchlen;
  NODID node_c, node_d, node_n, p;
  NODEID /*INTER*/ newlink = NODEID_NULL;
  int beta_st, beta_len, rest_st, rest_len, tail_st /* , tail_len */;
  NODEID node_tmp;

  stb_init_before_single_string(str, len, strn);
  i = 0;

  root = stb_create_root();
  stb_create_leaf(root.inter, i, strn);
  node_c = root;
  beta_len = 0;
  beta_st = 1;

  for (i = 1; i < len; i++) {
    if (beta_len > 0) {
      node_d = stb_mcc_rescan(node_c, str, beta_st, beta_len,
			      &matchlen, &substepb_offset, &parent);
      /* assert(matchlen == beta_len) */
      rest_st = beta_st + matchlen;
      rest_len = len - rest_st;
      if (substepb_offset != 0) {
	node_n = stb_create_internode(parent, node_d, substepb_offset);
	node_d = node_n;
	p = parent;
	matchlen = 0;
      }
    } else {
      node_d = node_c;
      rest_st = beta_st;
      rest_len = len - rest_st;
      substepb_offset = 0;
    }

    /* create new suffix link if needed */
    if (newlink != NODEID_NULL) {
      m_NODE_setmember_suffixlink(newlink, node_d);
    }

    if (substepb_offset != 0) {
      newlink = node_n;
      beta_st = rest_st - substepb_offset;
      beta_len = substepb_offset;
    } else {
      node_n = stb_mcc_scan(node_d, str, rest_st, rest_len,
		       &matchlen, &offset, &parent);
      if (offset != 0) {
	node_n = stb_create_internode(parent, node_n, offset);
	p = parent;
	newlink = node_n;
      } else {
	p = node_n;
	newlink = NODEID_NULL;
      }
      beta_st = rest_st + matchlen - offset;
      beta_len = offset;
    }

    tail_st = rest_st + matchlen;
    /* tail_len = rest_len - matchlen; */
    stb_create_leaf(node_n, tail_st, strn);

    node_tmp = m_NODE_getmember_suffixlink(p);
    if (node_tmp != NODEID_NULL) {
      node_c = node_tmp;
    } else {
      /* p is root */
      node_c = root;
      beta_st += 1;
      beta_len -= 1;
    }
  } /* for */

  return root;
} /* end of func */

NODEID stb_mcc_scan(NODEID sroot, uchar *str, int st, int len, 
 int *matchlen, int *offset, NODEID *parent)
/* matchlen, offset, parent are return values */
{
  NODEID cnode, tree;
  int mlen, cnode_len;
  uchar *s, *s_tree;
  int i;

  *matchlen = 0;
  *offset = 0;
  *parent = sroot;

  tree = sroot;
  mlen = 0;
  s = str + st;

  while (1) {
    if (mlen == len) {
      cnode = NODEID_NULL;
    } else {
      cnode = stb_mcc_find_child(tree, str, *s);
    }
    if (cnode == NODEID_NULL) {
      *offset = 0;
      *parent = tree;
      *matchlen = mlen;
      return tree;
    }

    s++;
    mlen++;
    cnode_len = m_stb_node_getlen(cnode);
    s_tree = str + cnode.leaf->s + 1;

    for (i = 1; i < cnode_len; i++) {
      if (mlen == len || *s++ != *s_tree++) {
	*offset = i;
	*parent = tree;
	*matchlen = mlen;
	return cnode;
      } else mlen++;
    } /* for */

    tree = cnode;
    /* assert(m_stb_self_is(tree) == NODE_IS_INTER) */

  } /* while */

  return sroot;
} /* end of func */


NODEID stb_mcc_rescan(NODEID sroot, uchar *str, int st, int len, 
 int *matchlen, int *offset, NODEID *parent)
/* matchlen, offset, parent are return values */
{
  NODEID cnode, tree;
  int mlen, cnode_len;
  uchar *s;

  *matchlen = 0;
  *offset = 0;
  *parent = sroot;

  tree = sroot;
  mlen = 0;
  s = str + st;

  while (1) {
    if (mlen == len) {
      cnode = NODEID_NULL;
    } else {
      cnode = stb_mcc_find_child(tree.inter, str, *s);
    }
    if (cnode == NODEID_NULL) {
      *offset = 0;
      *parent = tree;
      *matchlen = mlen;
      return tree;
    }

    cnode_len = m_stb_node_getlen(cnode);
    if (len - mlen >= cnode_len) {
      s += cnode_len;
      mlen += cnode_len;
    } else {
      *offset = len - mlen;
      *parent = tree;
      *matchlen = len;
      return cnode;
    }

    tree = cnode;
    /* assert(m_stb_self_is(tree) == NODE_IS_INTER) */
  } /* while */

  return sroot;
} /* end of func */
#endif /* MCCREIGHT */

/* ******************************************************************** */
NODEID stb_find_child(NODEID /*INTER*/ p, uchar ch)
{
  NODEID c;
#if STAT_VERBOSE
  g_count_findchild_call += 1;
  g_count_findchild += 1;
#endif /* STAT_VERBOSE */

  c = m_NODE_getmember_child(p);
  while (c != NODEID_NULL) {
    if (*(m_stb_node_getstr(c)) == ch) return c;
    c = m_NODE_getmember_neighbor(c);
#if STAT_VERBOSE
    g_count_findchild += 1;
#endif /* STAT_VERBOSE */
  }

  c = NODEID_NULL;
  return c;
} /* end of func */

NODEID stb_scan(NODEID sroot, int strno, int st, int len, 
 int *matchlen, int *offset, NODEID *parent)
/* matchlen, offset, parent are return values */
{
  NODEID cnode, tree;
  int mlen, cnode_len;
  uchar *s, *s_tree;
  int i;

  *matchlen = 0;
  *offset = 0;
  *parent = sroot;

  tree = sroot;
  mlen = 0;
  s = m_stb_getwholestring(strno) + st;

  while (1) {
    if (mlen >= len || m_stb_self_is(tree) != NODE_IS_INTER) {
      cnode = NODEID_NULL;
    } else {
      cnode = stb_find_child(tree, *s);
    }
    if (cnode == NODEID_NULL) {
      *offset = 0;
      *parent = tree;
      *matchlen = mlen;
      return tree;
    }

    cnode_len = m_stb_node_getlen(cnode);
    s_tree = m_stb_node_getstr(cnode) + 1;
#if 1
    s++;
    mlen++;
#else /* 1 */
    if (cnode_len > 0) {
      s++;
      mlen++;
    }
#endif

    for (i = 1; i < cnode_len; i++) {
      if (mlen == len || *s++ != *s_tree++) {
	*offset = i;
	*parent = tree;
	*matchlen = mlen;
	return cnode;
      } else mlen++;
    } /* for */

    tree = cnode;
  } /* while */

  return sroot;
} /* end of func */

NODEID stb_rescan(NODEID sroot, int strno, int st, int len, 
 int *matchlen, int *offset, NODEID *parent)
/* matchlen, offset, parent are return values */
{
  NODEID cnode, tree;
  int mlen, cnode_len;
  uchar *s;

  *matchlen = 0;
  *offset = 0;
  *parent = sroot;

  tree = sroot;
  mlen = 0;
  s = m_stb_getwholestring(strno) + st;

  while (1) {
    if (mlen >= len) { /* insted of == because safety */
      cnode = NODEID_NULL;
    } else {
      cnode = stb_find_child(tree, *s);
    }
    if (cnode == NODEID_NULL) {
      *offset = 0;
      *parent = tree;
      *matchlen = mlen;
      return tree;
    }

    cnode_len = m_stb_node_getlen(cnode);
    if (len - mlen >= cnode_len) {
      s += cnode_len;
      mlen += cnode_len;
    } else {
      *offset = len - mlen;
      *parent = tree;
      *matchlen = len;
      return cnode;
    }

    tree = cnode;
    /* assert(m_stb_self_is(tree) == NODE_IS_INTER) */
  } /* while */

  return sroot;
} /* end of func */
/* ******************************************************************** */
/* ******************************************************************** */
int stb_ukk_singlephase(NODEID root, uchar *str, int edpos, int leafs,
 int strno, NODEID *r3node, int *r3pos)
/* r3node, r3pos are return values */
{
  NODEID tree, dnode, ddnode;
  NODEID newnode;
  int i;
  NODEID parent;
  int offset, matchlen;
  int pos;
  NODEID node_tmp;

  newnode = NODEID_NULL;
  if ((*r3node) == NODEID_NULL) {
    tree = root;
    pos = leafs;
  } else {
    pos = *r3pos;
    tree = *r3node;
  }
  (*r3node) = NODEID_NULL;

  for (i = leafs; ; ) {
    dnode = stb_rescan(tree, strno, pos, edpos - pos,
			   &matchlen, &offset, &parent);
    if (offset == 0) {
      /* create new suffix link if needed */
      if (newnode != NODEID_NULL) {
	m_NODE_setmember_suffixlink(newnode, dnode);
      }
      ddnode = stb_find_child(dnode, str[edpos]);
      if (ddnode != NODEID_NULL) {
	/* rule 3 */
	*r3node = dnode;
	*r3pos = edpos;
	break; /* for */
      } else {
	ddnode = dnode;
	newnode = NODEID_NULL;
      }
    } else { /* offset != 0 */
      if (*(m_stb_node_getstr(dnode) + offset) == str[edpos]) {
	/* rule 3 */
	*r3node = parent;
	*r3pos = edpos - offset;
	break;
      } else {
	ddnode = stb_create_internode(parent, dnode, offset, pos + matchlen - offset);
	/* create new suffix link if needed */
	if (newnode != NODEID_NULL) {
	  m_NODE_setmember_suffixlink(newnode, ddnode);
	}
	dnode = parent;
	newnode = ddnode;
      }
    } /* if (offset == 0) */

    /* rule 2 */
    /* create a leaf */
    stb_create_leaf(ddnode, edpos, strno);
    
    /* prepare next step */
    if (i++ >= edpos) break; /* for */

    node_tmp = m_NODE_getmember_suffixlink(dnode);
    if (node_tmp != NODEID_NULL) {
      tree = node_tmp;
      pos = edpos - offset;
    } else {
      tree = root;
      pos = i;
    }
  } /* for */

  return i; /* new number of leafs */
} /* end of func */

NODEID st_buildsuffixtree_ukkonen(uchar *str, int len)
/* len should contain length of last '\0' */
{
  const int strno = 0;
  NODEID root;
  NODEID r3node;
  int r3pos = 0;
  int leafs = 0;
  int i;

  r3node = NODEID_NULL;
  stb_init_before_single_string(str, len, strno);
  root = stb_create_root();

  for (i = 0; i < len; i++) {
    leafs = stb_ukk_singlephase(root, str, i, leafs, strno,
				&r3node, &r3pos);
  }
  stb_init_after_single_string(strno);

  return root;
} /* end of func */

NODEID st_generalizedsuffixtree_ukkonen(NODEID root, uchar *str, int len)
/* len should contain length of last '\0' */
{
  int strno; 
  NODEID dnode;
  NODEID r3node;
  int r3pos = 0;
  int leafs = 0;
  int i;
  int matchlen, offset;
  NODEID parent;

  strno = ++g_currentstrno;
  r3node = NODEID_NULL;
  stb_init_before_single_string(str, len, strno);

  dnode = stb_scan(root, strno, 0, len - 1,
		     &matchlen, &offset, &parent);

  for (i = matchlen; i < len; i++) {
    leafs = stb_ukk_singlephase(root, str, i, leafs, strno,
				&r3node, &r3pos);
  }

  if (r3node != NODEID_NULL) {
    stb_ukk_generalized_lastphase(root, r3node, r3pos,
				  leafs, str, strno, len - 1);
  }

  stb_init_after_single_string(strno);
  return root;
} /* end of func */

void stb_ukk_generalized_lastphase(NODEID root, NODEID r3node, int r3pos,
 int leafs, uchar *str, int strno, int edpos)
{
  int i;
  NODEID dnode, ddnode;
  NODEID parent;
  int matchlen, offset;
  NODEID /*INTER*/ newlinknode;
  NODEID newnode;
  NODEID node_tmp;

  newlinknode = NODEID_NULL;
  for (i = leafs; ; ) {
    dnode = stb_rescan(r3node, strno, r3pos, edpos - r3pos,
		       &matchlen, &offset, &parent);
    /* assert(dnode != NODEID_NULL) */
    if (offset != 0) {
      newnode = stb_create_internode(parent, dnode, offset, r3pos + matchlen - offset);
      dnode = newnode;
#if STAT
      g_count_internode_pseudo += 1;
#endif /* STAT */
    } else {
      newnode = NODEID_NULL;
    }
    if (newlinknode != NODEID_NULL) {
      m_NODE_setmember_suffixlink(newlinknode, dnode);
    }
    newlinknode = newnode;
    ddnode = stb_create_leaf(dnode, edpos, strno);
#if STAT
    g_count_leaf_same += 1;
#endif /* STAT */
    
    if (i++ >= edpos) break; /* for */

    node_tmp = m_NODE_getmember_suffixlink(parent);
    if (node_tmp != NODEID_NULL) {
      r3node = node_tmp;
      r3pos = edpos - offset;
    } else {
      r3node = root;
      r3pos = i;
    }
  } /* for */

  return;
} /* end of func */



/* ******************************************************************** */
#define STBX_STACKUNIT 1000
NODEID *g_stbx_stack_node = NULL;
NODEID *g_stbx_stack_node_orig = NULL;
NODEID *g_stbx_stack_node_overflow;
int g_stbx_stack_node_size = 0;

uchar *g_stbx_stack_lc = NULL;
uchar *g_stbx_stack_lc_orig = NULL;
uchar *g_stbx_stack_lc_overflow;
int g_stbx_stack_lc_size = 0;

#define macro_stbx_stack_extend(name, type, unit) \
do { \
  ptrdiff_t tmp = 0; \
  /* if (name != NULL && name ## _orig != NULL) */ \
      tmp = name - name ## _orig; \
  name ## _size += (unit); \
  name ## _orig = my_realloc(name ## _orig, name ## _size * sizeof(type)); \
  name = name ## _orig + tmp; \
  name ## _overflow = name ## _orig + name ## _size; \
} while (0)

#define macro_stbx_stack_free(name) \
( \
 my_free(name ## _orig), \
 name ## _orig = NULL, \
 name = NULL, \
 name ## _overflow = NULL, \
 name ## _size = 0 \
)

#define macro_stbx_stack_reset(name) \
(name = name ## _orig)

#define macro_stbx_stack_push(name, type, data) \
do { \
  if (name ## _overflow <= name) { \
    macro_stbx_stack_extend(name, type, STBX_STACKUNIT); \
  } \
  * name ++ = (data); \
} while (0)

#define macro_stbx_stack_push_mext(name, type, data, name2, type2) \
do { \
  if (name ## _overflow <= name) { \
    macro_stbx_stack_extend(name, type, STBX_STACKUNIT); \
    macro_stbx_stack_extend(name2, type2, STBX_STACKUNIT); \
  } \
  * name ++ = (data); \
} while (0)

#define macro_stbx_stack_empty(name) (name <= name ## _orig)

#if DEBUG_MEM
#define macro_stbx_stack_push_nc(name, data) \
(name < name ## _overflow ? (* name ++ = (data)) \
 : (fprintf(stderr, "Stack overflow!\n"), (*name ++ = (data))))
#define macro_stbx_stack_pop_nc(name) \
(name > name ## _orig ? *(-- name) \
: (fprintf(stderr, "Stack underflow!\n"), *(-- name)))
#define macro_stbx_stack_pop(name) macro_stbx_stack_pop_nc(name)
#else
#define macro_stbx_stack_push_nc(name, data) (* name ++ = (data))
#define macro_stbx_stack_pop_nc(name) (*(-- name))
#define macro_stbx_stack_pop(name) macro_stbx_stack_pop_nc(name)
#endif /* DEBUG_MEM */

void stbx_stack_init(void)
{
  macro_stbx_stack_extend(g_stbx_stack_node, NODEID, STBX_STACKUNIT);
  macro_stbx_stack_extend(g_stbx_stack_lc, uchar, STBX_STACKUNIT);
} /* end of func */

void stbx_stack_free(void)
{
  macro_stbx_stack_free(g_stbx_stack_node);
  macro_stbx_stack_free(g_stbx_stack_lc);
} /* end of func */

int stbx_stack_empty(void)
{
  return macro_stbx_stack_empty(g_stbx_stack_node);
} /* end of func */

NODEID stbx_stack_node_pop(void)
{
  return macro_stbx_stack_pop(g_stbx_stack_node);
} /* end of func */

void stbx_stack_node_push(NODEID node)
{
  macro_stbx_stack_push_mext(g_stbx_stack_node, NODEID, node,
			     g_stbx_stack_lc, uchar);
} /* end of func */

#define m_stbx_stack_lc_pop() \
macro_stbx_stack_pop_nc(g_stbx_stack_lc)
#define m_stbx_stack_lc_push(lc) \
macro_stbx_stack_push_nc(g_stbx_stack_lc, lc)

int stb_mark_leftdiversity(NODEID root, int *maxlen_internode)
{
  NODEID node = root;
  uchar lc = LEFTCHAR_NULL;
  uchar lc_tmp = LEFTCHAR_NULL;
  int maxdepth = 0, currentdepth = 0;
  int currentlen = 0, maxlen = 0;

  stbx_stack_init();

  while (1) {
    if (node == NODEID_NULL) {
      if (currentdepth > maxdepth) maxdepth = currentdepth;
      currentdepth -= 1;
      node = stbx_stack_node_pop();
      m_NODE_setmember_leftchar(node, lc);
      lc_tmp = m_stbx_stack_lc_pop();
      if (stbx_stack_empty()) break;
      if (lc_tmp != LEFTCHAR_NULL && lc_tmp != lc) {
	lc = END_OF_STR;
      }
      currentlen -= m_stb_node_getlen(node);
      /* next node */
      node = m_NODE_getmember_neighbor(node);
    } else if (m_stb_self_is(node) == NODE_IS_INTER) {
      currentlen += m_stb_node_getlen_inter(node);
      if (currentlen > maxlen) maxlen = currentlen;
      stbx_stack_node_push(node);
      m_stbx_stack_lc_push(lc);
      lc = LEFTCHAR_NULL;
      currentdepth += 1;
      /* next node */
      node = m_NODE_getmember_child(node);
    } else { /* node is LEAF */
      /* check left_diversity */
      if (lc == LEFTCHAR_NULL) {
	lc = m_NODE_getmember_leftchar(node);
      } else if (lc != m_NODE_getmember_leftchar(node)) {
	lc = END_OF_STR; /* means leftdiversity */
      }
      /* next node */
      node = m_NODE_getmember_neighbor(node);
    }
  } /* while */

  stbx_stack_free();
  *maxlen_internode = maxlen;
  return maxdepth;
} /* end of func */

NODEID /*LEAF*/ *g_stbo_stack_leaf = NULL;
NODEID /*LEAF*/ *g_stbo_stack_leaf_orig = NULL;
NODEID /*LEAF*/ *g_stbo_stack_leaf_overflow;
int g_stbo_stack_leaf_size = 0;

void stbo_stack_leaf_init(void)
{
  int i, l;

  l = 0;
  for (i = 0; i <= g_currentstrno; i++) {
    l += m_stb_getwholestrlen(i);
  }

  l /= 3;
  macro_stbx_stack_extend(g_stbo_stack_leaf, NODEID, l);
  return;
} /* end of func */

void stbo_stack_leaf_free(void)
{
  macro_stbx_stack_free(g_stbo_stack_leaf);
} /* end of func */

void stbo_stack_leaf_reset(void)
{
  macro_stbx_stack_reset(g_stbo_stack_leaf);
  return;
} /* end of func */

void stbo_stack_leaf_push(NODEID /*LEAF*/ data)
{
  macro_stbx_stack_push(g_stbo_stack_leaf, NODEID, data);
} /* end of func */

NODEID /*LEAF*/ stbo_stack_leaf_pop(void)
{
  return macro_stbx_stack_pop(g_stbo_stack_leaf);
} /* end of func */

void stbo_stack_node_init(int maxdepth)
{
  maxdepth += 2;
  macro_stbx_stack_extend(g_stbx_stack_node, NODEID, maxdepth);

  return;
} /* end of func */

void stbo_stack_node_reset(void)
{
  macro_stbx_stack_reset(g_stbx_stack_node);
  return;
} /* end of func */

void stbo_stack_node_free(void)
{
  macro_stbx_stack_free(g_stbx_stack_node);
} /* end of func */

int stbo_stack_node_empty(void)
{
  return macro_stbx_stack_empty(g_stbx_stack_node);
} /* end of func */

#define m_stbo_stack_node_pop() \
macro_stbx_stack_pop_nc(g_stbx_stack_node)
#define m_stbo_stack_node_push(data) \
macro_stbx_stack_push_nc(g_stbx_stack_node, data)

int *g_stbo_stack_int = NULL;
int *g_stbo_stack_int_orig = NULL;
int *g_stbo_stack_int_overflow;
int g_stbo_stack_int_size = 0;

void stbo_stack_int_init(int maxdepth)
{
  maxdepth += 2;
#if CALC_SD
  maxdepth *= 2;
#endif /* CALC_SD */
  macro_stbx_stack_extend(g_stbo_stack_int, int, maxdepth);

  return;
} /* end of func */

void stbo_stack_int_free(void)
{
  macro_stbx_stack_free(g_stbo_stack_int);
} /* end of func */

void stbo_stack_int_reset(void)
{
  macro_stbx_stack_reset(g_stbo_stack_int);
  return;
} /* end of func */

#define m_stbo_stack_int_pop() \
macro_stbx_stack_pop_nc(g_stbo_stack_int)
#define m_stbo_stack_int_push(data) \
macro_stbx_stack_push_nc(g_stbo_stack_int, data)

#if CALC_VARIETY
int *g_stbo_varray_stack_orig = NULL;
int *g_stbo_varray_stack = NULL;

void stbo_varray_allocstack(int maxdepth, int grps)
{
  maxdepth += 2;
  g_stbo_varray_stack_orig = my_malloc(maxdepth * grps * sizeof(int));
  g_stbo_varray_stack = g_stbo_varray_stack_orig;
  return;
} /* end of func */

void stbo_varray_freestack(void)
{
  my_free(g_stbo_varray_stack_orig);
  g_stbo_varray_stack_orig = NULL;
  g_stbo_varray_stack = NULL;
  return;
} /* end of func */

void stbo_varray_resetstack(void)
{
  g_stbo_varray_stack = g_stbo_varray_stack_orig;
  return;
} /* end of func */

void stbo_varray_push(int *varray, int grps)
{
  memcpy(g_stbo_varray_stack, varray, grps * sizeof(int));
  g_stbo_varray_stack += grps;
  return;
} /* end of func */

void stbo_varray_init(int *varray, int grps)
{
  while (grps-- > 0) {
    *varray++ = 0;
  }
  return;
} /* end of func */

void stbo_varray_popandadd(int *varray, int grps)
{

  varray += grps;
  while (grps-- > 0) {
#if CALCV_FAST
    *(--varray) |= *(--g_stbo_varray_stack);
#else /* CALCV_FAST */
    *(--varray) += *(--g_stbo_varray_stack);
#endif /* CALCV_FAST */
  }
  return;
} /* end of func */

int stbo_varray_calc(int *varray, int grps)
{
  int v = 0;

  varray += grps;
  while (grps-- > 0) {
#if CALCV_FAST
    v += *(--varray);
#else /* CALCV_FAST */
    v += ((*(--varray) > 0) ? 1 : 0);
#endif /* CALCV_FAST */
  }
  return v;
} /* end of func */
#endif /* CALC_VARIETY */

#if SHOW_SEQ
void stbo_seqshow(int strno, int st, int len, FILE *fpo)
/* top of string is 0 */
{
  uchar *str;

  if (g_opt_max_seqshowlen >= len) {
    str = m_stb_getwholestring(strno) + st;
    fputc(' ', fpo);
    while (len-- > 0) {
      fputc(*str++, fpo);
    }
  }

  return;
} /* end of func */
#endif /* SHOW_SEQ */

void stbo_outfasta(int strno, int st, int len, FILE *fp_fst)
/* top of string is 0 */
{
  uchar *str;
  int c = 0;

  str = m_stb_getwholestring(strno) + st;
  while (len-- > 0) {
    fputc(*str++, fp_fst);
    if (++c >= g_opt_fastawidth) {
      c = 0;
      fputc('\n', fp_fst);
    }
  }
  if (c != 0) fputc('\n', fp_fst);

  return;
} /* end of func */

void stbo_output(int num, int currentlen, int variety, 
 int *counttable, int *flagtable, int *strnotable,
 int *offsettable, int *offsettable_c,
 int groupmode, enum OutputMode output_mode,
 int counter, FILE *fpo, FILE *fp_fst)
{
  NODEID /*LEAF*/ *stack;
  NODEID /*LEAF*/ data;
  int strno, leafno;
  int i;
  const static char *idstr = "No_";

#if CALC_VARIETY
  fprintf(fpo, "n=%d L=%d v=%d ---- %s%d", num, currentlen, variety,
	  idstr, counter);
  if (fp_fst != NULL) 
    fprintf(fp_fst, ">%s%d n=%d L=%d v=%d\n", idstr, counter, 
	    num, currentlen, variety);
#else /* CALC_VARIETY */
  fprintf(fpo, "n=%d L=%d ---- %s%d", num, currentlen,
	  idstr, counter);
  if (fp_fst != NULL) 
    fprintf(fp_fst, ">%s%d n=%d L=%d\n", idstr, counter, 
	    num, currentlen);
#endif /* CALC_VARIETY */

  stack = g_stbo_stack_leaf;
  data = *(stack - 1);
  strno = m_NODE_getmember_strno(data);
  leafno = stb_get_leafno(data);
  if (fp_fst != NULL) stbo_outfasta(strno, leafno, currentlen, fp_fst);
#if SHOW_SEQ
  stbo_seqshow(strno, leafno, currentlen, fpo);
#endif /* SHOW_SEQ */
  fprintf(fpo, "\n");
  for (i = num ; i > 0; i--) {
    data = *(--stack);
    strno = m_NODE_getmember_strno(data);
    leafno = stb_get_leafno(data);
    if (flagtable[strno]) {
      fprintf(fpo, "%d c%d ( %dc %d )",
	      strnotable[strno], 
	      offsettable_c[strno] - leafno - currentlen,
	      strnotable[strno],
	      offsettable[strno] + leafno
	      );
    } else {
      fprintf(fpo, "%d %d", strnotable[strno],
	      offsettable[strno] + leafno);
    }
    if (groupmode) {
      fprintf(fpo, " ( g=%d )\n", counttable[strno] + 1);
    } else {
      fprintf(fpo, "\n");
    }
  } /* for */

  if (output_mode == output_mixed) {
#if CALC_VARIETY
    fprintf(fpo, ">%s%d n=%d L=%d v=%d\n", idstr, counter, 
	    num, currentlen, variety);
#else /* CALC_VARIETY */
    fprintf(fpo, ">%s%d n=%d L=%d\n", idstr, counter, 
	    num, currentlen);
#endif /* CALC_VARIETY */
    stbo_outfasta(strno, leafno, currentlen, fpo);
    fprintf(fpo, "> \n");
  }
  return;
} /* end of func */

int stbo_traceall(NODEID /*INTER*/ root, int minlen_out, int minlen_calc,
 int min_variety, int min_ncount,
 int *stat_c, int *stat_v, 
 int *counttable, int *flagtable, int *strnotable,
 int *offsettable, int *offsettable_c,
 int strs, int grps, int groupmode,
 enum OutputMode output_mode,
 int counter, FILE *fpo, FILE *fp_fst)
{
  NODEID node;
  int currentlen;
  int *varray;
  int strno;

  int variety = 0;
  int ns = 0;
#if CALC_SD
  int sd = STRNO_NULL;
  int sd_tmp = STRNO_NULL;
#endif /* CALC_SD */

#if CALC_VARIETY
  varray = my_calloc(grps, sizeof(int));
#endif /* CALC_VARIETY */
  currentlen = 0;
  node = root;
  while (1) {
    if (node == NODEID_NULL) {
      node = m_stbo_stack_node_pop();
      if (currentlen >= minlen_calc
	  && ns >= min_ncount
	  && m_NODE_getmember_leftchar(node) == END_OF_STR /* left diversity */
#if CALC_SD
	  && sd == STRNO_DIVERSITY
#endif /* CALC_SD */
	  ) {
#if CALC_VARIETY
	variety = stbo_varray_calc(varray, grps);
	if (variety < min_variety) {
	  goto SKIP; /* very dirty code!!! */
	}
#endif /* CALC_VARIETY */
	stat_v[currentlen] += 1;
	stat_c[currentlen] += ns;
	if (currentlen >= minlen_out) {
	  stbo_output(ns, currentlen, variety,
		      counttable, flagtable, strnotable,
		      offsettable, offsettable_c,
		      groupmode, output_mode,
		      counter++, fpo, fp_fst);
	}
      } /* if(currentlen...) */
    SKIP:
      if (stbo_stack_node_empty()) break;
      ns += m_stbo_stack_int_pop();
#if CALC_SD
      sd_tmp = m_stbo_stack_int_pop();
      if (sd == STRNO_NULL) {
	sd = sd_tmp;
      } else if (sd_tmp != STRNO_NULL && sd_tmp != sd) {
	sd = STRNO_DIVERSITY;
      }
#endif /* CALC_SD */
#if CALC_VARIETY
      stbo_varray_popandadd(varray, grps);
#endif /* CALC_VARIETY */
      currentlen -= m_stb_node_getlen(node);
      /* next node */
      node = m_NODE_getmember_neighbor(node);
    } else if (m_stb_self_is(node) == NODE_IS_INTER) {
      currentlen += m_stb_node_getlen_inter(node);
      m_stbo_stack_node_push(node);
#if CALC_VARIETY
      stbo_varray_push(varray, grps);
      stbo_varray_init(varray, grps);
#endif /* CALC_VARIETY */
#if CALC_SD
      m_stbo_stack_int_push(sd);
      sd = STRNO_NULL;
#endif /* CALC_SD */
      m_stbo_stack_int_push(ns);
      ns = 0;
      /* next node */
      node = m_NODE_getmember_child(node);
    } else { /* node is LEAF */
      stbo_stack_leaf_push(node);
      ns += 1;
      strno = m_NODE_getmember_strno(node);
#if CALC_VARIETY
#if CALCV_FAST
      varray[counttable[strno]] |= 1;
#else /* CALCV_FAST */
      varray[counttable[strno]]++;
#endif /* CALCV_FAST */
#endif /* CALC_VARIETY */
#if CALC_SD
      if (sd == STRNO_NULL) {
	sd = strno;
      } else if (sd != strno) {
	sd = STRNO_DIVERSITY;
      }
#endif /* CALC_SD */
      /* next node */
      node = m_NODE_getmember_neighbor(node);
    }
  } /* while */

#if CALC_VARIETY
  my_free(varray);
#endif /* CALC_VARIETY */
  return counter;
} /* end of func */

void stb_output_identical_string(NODEID root, int maxdepth, int maxlen,
 int minlen_out, int minlen_stat, int min_variety, int min_ncount,
 int *stat_c, int *stat_v,
 int *counttable, int *flagtable, int *strnotable,
 int *offsettable, int *offsettable_c,
 int strs, int grps,
 int groupmode, enum OutputMode output_mode,
 FILE *fpo, FILE *fp_fst)
{
  NODEID node;
  int minlen_calc;
  int counter = 1;

  stbo_stack_node_init(maxdepth);
  stbo_stack_int_init(maxdepth);
  stbo_stack_leaf_init();
  stbo_varray_allocstack(maxdepth, strs);

  minlen_calc = ((minlen_stat < minlen_out) ? minlen_stat : minlen_out);
  for (node = m_NODE_getmember_child(root); node != NODEID_NULL;) {
    if (m_stb_self_is(node) == NODE_IS_INTER) {
      counter = stbo_traceall(node,
			      minlen_out, minlen_calc, min_variety, min_ncount,
			      stat_c, stat_v, 
			      counttable, flagtable, strnotable,
			      offsettable, offsettable_c,
			      strs, grps, groupmode, output_mode,
			      counter, fpo, fp_fst);
    }
    node = m_NODE_getmember_neighbor(node);
    stbo_stack_node_reset();
    stbo_stack_int_reset();
    stbo_stack_leaf_reset();
    stbo_varray_resetstack();
  }

  stbo_varray_freestack();
  stbo_stack_leaf_free();
  stbo_stack_int_free();
  stbo_stack_node_free();

  return;
} /* end of func */


/* ******************************************************************** */
/* memory management */
/* NODEID 
   bit  0 -- 11 : (2^12=4096)    
   bit 12 -- 30 : (2^19=524288) 
   bit 31       : (2^1 =2)      leaf(1) or intenal(0)
*/

#define G_NODEID_MEMUNIT_FLAGBIT 31
#define G_NODEID_MEMUNIT_MASK_F  ((NODEID)0x80000000)
#define G_NODEID_MEMUNIT_MASK_N  ((NODEID)0x7fffffff)

#define m_NODEID_get_MEMUNIT_flag(id) (id & G_NODEID_MEMUNIT_MASK_F)

/* NODEID of leaf */
#define G_NODEID_MEMUNIT_BITS 12 /* 2^12 = 4096 */

#define G_NODEID_MEMUNIT_OFFSETS ((NODEID)0x00001000)
#define G_NODEID_MEMUNIT_MASK_O  ((NODEID)0x00000fff)

#define m_NODEID_get_MEMUNIT_N(id) \
 ((id & G_NODEID_MEMUNIT_MASK_N) >> G_NODEID_MEMUNIT_BITS)
#define m_NODEID_get_MEMUNIT_OFFSET(id) (id & G_NODEID_MEMUNIT_MASK_O)

/* NODEID of internode */
#define G_NODEID_INTER_MEMUNIT_BITS 12 /* 2^12 = 4096 */

#define G_NODEID_INTER_MEMUNIT_OFFSETS ((NODEID)0x00001000)
#define G_NODEID_INTER_MEMUNIT_MASK_O  ((NODEID)0x00000fff)

#define m_NODEID_INTER_get_MEMUNIT_N(id) \
 ((id & G_NODEID_MEMUNIT_MASK_N) >> G_NODEID_INTER_MEMUNIT_BITS)
#define m_NODEID_INTER_get_MEMUNIT_OFFSET(id) \
 (id & G_NODEID_INTER_MEMUNIT_MASK_O)



struct st_MEMUNIT {
  STRNO_T strno;
  NODEPTR orig;
  NODEID str_base_NODEID;
  NODEID *orig_suffixlink;
  unsigned char *orig_leftchar;
};

struct st_MEMUNIT_admin {
  struct st_MEMUNIT *memunit;
  NODEID units;
  NODEID unit_current;
  NODEID offset_current;
  NODEID id_current;
};

struct st_MEMUNIT_admin g_memunit_admin_inter;
struct st_MEMUNIT_admin g_memunit_admin_leaf;

struct leafnode **g_store_leaf;
unsigned char **g_store_leaf_leftchar;
NODEID g_store_leaf_offset;

struct internode **g_store_inter;
NODEID **g_store_inter_suffixlink;
NODEID g_store_inter_offset;
unsigned char **g_store_inter_leftchar;

int g_num_of_strings = 1;
int *g_strlen;
uchar **g_str;
int g_currentstrno = 0;

uchar *g_currentleftchar;
uchar g_nextleftchar;

static void st_MEMUNIT_set_strno(struct st_MEMUNIT *muP, STRNO_T strno,
 NODEID str_base_id)
{
  muP->strno = strno;
  muP->str_base_NODEID = str_base_id;
  return;
} /* end of func */

#if 0
static void st_newalloc_MEMUNIT_inter(struct st_MEMUNIT *muP, int strno,
 NODEID str_base_id)
{
  muP->strno = strno;
  muP->str_base_NODEID = str_base_id;
  muP->orig.inter = my_calloc(G_NODEID_INTER_MEMUNIT_OFFSETS, 
			      sizeof(struct internode));
  muP->orig_suffixlink = my_malloc(G_NODEID_INTER_MEMUNIT_OFFSETS * 
				   sizeof(NODEID));
  muP->orig_leftchar = my_calloc(G_NODEID_INTER_MEMUNIT_OFFSETS, 
				 sizeof(unsigned char));
#if STAT
  g_totalmem += G_NODEID_INTER_MEMUNIT_OFFSETS * sizeof(struct internode);
  g_totalmem += G_NODEID_INTER_MEMUNIT_OFFSETS * sizeof(NODEID);
  g_totalmem += G_NODEID_INTER_MEMUNIT_OFFSETS * sizeof(unsigned char);
#endif
  return;
} /* end of func */
#endif /* 0 */

static void init_st_MEMUNIT_admin_inter(int units)
{
  struct st_MEMUNIT_admin *smuaP = &g_memunit_admin_inter;

  smuaP->memunit = my_calloc(units, sizeof(struct st_MEMUNIT));
#if STAT
  g_totalmem += (units * sizeof(struct st_MEMUNIT));
#endif
  smuaP->units = units;
  smuaP->unit_current = 0;
  smuaP->offset_current = 0;

  /* st_newalloc_MEMUNIT_inter(&(smuaP->memunit[0]), 0, 0); */
  return;
} /* end of func */

static void init_st_MEMUNIT_admin_leaf(int units)
{
  struct st_MEMUNIT_admin *smuaP = &g_memunit_admin_leaf;

  smuaP->memunit = my_calloc(units, sizeof(struct st_MEMUNIT));
#if STAT
  g_totalmem += (units * sizeof(struct st_MEMUNIT));
#endif
  smuaP->units = units;
  smuaP->unit_current = 0;
  smuaP->offset_current = 0;

  return;
} /* end of func */

void stb_initialize_memadm_part1(int num_of_strings)
{
  g_num_of_strings = num_of_strings;
  g_strlen = my_calloc(num_of_strings, sizeof(int));
  g_str = my_calloc(num_of_strings, sizeof(uchar *));
#if STAT
  g_totalmem += num_of_strings * sizeof(int);
  g_totalmem += num_of_strings * sizeof(uchar *);
#endif

  g_store_leaf = my_calloc(num_of_strings, sizeof(struct leafnode *));
  g_store_leaf_leftchar = 
    my_calloc(num_of_strings, sizeof(unsigned char *));
#if STAT
  g_totalmem += num_of_strings * sizeof(struct leafnode *);
  g_totalmem += num_of_strings * sizeof(unsigned char *);
#endif

  g_store_inter = my_calloc(num_of_strings, sizeof(struct internode *));
  g_store_inter_suffixlink = 
    my_calloc(num_of_strings, sizeof(NODEID *));
  g_store_inter_leftchar = 
    my_calloc(num_of_strings, sizeof(unsigned char *));
#if STAT
  g_totalmem += num_of_strings * sizeof(struct internode *);
  g_totalmem += num_of_strings * sizeof(NODEID *);
  g_totalmem += num_of_strings * sizeof(unsigned char *);
#endif

  return;
} /* end of func */

static int calc_required_units(int *array_strlen, int num_of_strings)
{
  int i;
  NODEID len;
  NODEID units = 0;

  for (i = 0; i < num_of_strings; i++) {
    len = array_strlen[i];
    units += (len / G_NODEID_MEMUNIT_OFFSETS);
    if(len % G_NODEID_MEMUNIT_OFFSETS > 0) units += 1;
  } /* for (i) */

  fprintf(stderr, "total %d MEMUNITs used\n", units);
  return units;
} /* end of func */

void stb_initialize_memadm_part2(int num_of_strings)
{
  NODEID units;

  units = calc_required_units(g_strlen, num_of_strings);
  init_st_MEMUNIT_admin_leaf(units);
  init_st_MEMUNIT_admin_inter(units);

  g_memunit_admin_inter.id_current = 0;
  g_memunit_admin_leaf.id_current = G_NODEID_MEMUNIT_MASK_F;

  return;
} /* end of func */

#if 0
static void st_resize_MEMUNIT_inter(struct st_MEMUNIT *muP, int newsize)
{
  muP->orig.inter = my_realloc(muP->orig.inter,
			       newsize * sizeof(struct internode));
  muP->orig_suffixlink = my_realloc(muP->orig_suffixlink, 
				    newsize * sizeof(NODEID));
  muP->orig_leftchar = my_realloc(muP->orig_leftchar, 
				  newsize * sizeof(unsigned char));
#if STAT
  g_totalmem -= G_NODEID_INTER_MEMUNIT_OFFSETS * sizeof(struct internode);
  g_totalmem -= G_NODEID_INTER_MEMUNIT_OFFSETS * sizeof(NODEID);
  g_totalmem -= G_NODEID_INTER_MEMUNIT_OFFSETS * sizeof(unsigned char);
  g_totalmem += newsize * sizeof(struct internode);
  g_totalmem += newsize * sizeof(NODEID);
  g_totalmem += newsize * sizeof(unsigned char);
#endif /* STAT */
  return;
} /* end of func */

static void st_change_to_next_MEMUNIT_inter(int strno)
{
  struct st_MEMUNIT_admin *smuaP = &g_memunit_admin_inter;
  NODEID cur = smuaP->unit_current;
  NODEID offs = smuaP->offset_current;

  if (offs > 0) {
    st_resize_MEMUNIT_inter(&(smuaP->memunit[cur]), offs);
    smuaP->offset_current = 0;
    smuaP->unit_current += 1;
    cur += 1;
    smuaP->id_current -= offs;
    smuaP->id_current += G_NODEID_MEMUNIT_OFFSETS;
    st_newalloc_MEMUNIT_inter(&(smuaP->memunit[cur]), strno,
			    smuaP->id_current);
  } else {
    st_MEMUNIT_set_strno(&(smuaP->memunit[cur]), strno, smuaP->id_current);
  }

  return;
} /* end of func */
#endif /* 0 */

static void st_change_to_next_MEMUNIT_leaf(int strno)
{
  struct st_MEMUNIT_admin *smuaP = &g_memunit_admin_leaf;
  NODEID cur = smuaP->unit_current;
  NODEID offs = smuaP->offset_current;

  if (offs > 0) {
    smuaP->offset_current = 0;
    smuaP->unit_current += 1;
    cur += 1;
    smuaP->id_current -= offs;
    smuaP->id_current += G_NODEID_MEMUNIT_OFFSETS;
  }
  smuaP->memunit[cur].orig.leaf = g_store_leaf[strno];
  smuaP->memunit[cur].orig_leftchar = g_store_leaf_leftchar[strno];
  st_MEMUNIT_set_strno(&(smuaP->memunit[cur]), strno, smuaP->id_current);
  return;
} /* end of func */

static void st_change_to_next_MEMUNIT_inter(int strno)
{
  struct st_MEMUNIT_admin *smuaP = &g_memunit_admin_inter;
  NODEID cur = smuaP->unit_current;
  NODEID offs = smuaP->offset_current;

  if (offs > 0) {
    smuaP->offset_current = 0;
    smuaP->unit_current += 1;
    cur += 1;
    smuaP->id_current -= offs;
    smuaP->id_current += G_NODEID_MEMUNIT_OFFSETS;
  }
  smuaP->memunit[cur].orig.inter = g_store_inter[strno];
  smuaP->memunit[cur].orig_suffixlink = g_store_inter_suffixlink[strno];
  smuaP->memunit[cur].orig_leftchar = g_store_inter_leftchar[strno];
  st_MEMUNIT_set_strno(&(smuaP->memunit[cur]), strno, smuaP->id_current);
  return;
} /* end of func */

void stb_init_before_single_string(uchar *str, int size, int strno)
{
  int i;

  g_store_leaf[strno] = my_malloc(size * sizeof(struct leafnode));
  g_store_leaf_leftchar[strno] = my_malloc(size * sizeof(unsigned char));
#if STAT
  g_totalmem += (size * sizeof(struct leafnode));
  g_totalmem += (size * sizeof(unsigned char));
#endif /* STAT */
  g_store_leaf_offset = 0;

  g_store_inter[strno] = my_malloc((size+1) * sizeof(struct internode));
  g_store_inter_suffixlink[strno] = my_malloc((size+1) * sizeof(NODEID));
  g_store_inter_leftchar[strno] = my_calloc((size+1), sizeof(unsigned char));

  for (i = 0; i < (size+1); i++) {
    g_store_inter_suffixlink[strno][i] = NODEID_NULL;
  }
#if STAT
  /* memory size is added after building tree */
#endif /* STAT */
  g_store_inter_offset = 0;

  st_change_to_next_MEMUNIT_inter(strno);
  st_change_to_next_MEMUNIT_leaf(strno);

  g_strlen[strno] = size;
  g_str[strno] = str;

  g_currentleftchar = str;
  g_nextleftchar = 0;

  return;
} /* end of func */

void stb_init_after_single_string(int strno)
{
  NODEID offset;
  struct st_MEMUNIT_admin *smuaP = &g_memunit_admin_inter;
  struct internode *p;
  NODEID *p_s;
  unsigned char *p_l;
  NODEID i;

  /* return; */
  offset = g_store_inter_offset + smuaP->offset_current;
  g_store_inter[strno] =
    my_realloc(g_store_inter[strno], offset * sizeof(struct internode));
  g_store_inter_suffixlink[strno] =
    my_realloc(g_store_inter_suffixlink[strno], offset * sizeof(NODEID));
  g_store_inter_leftchar[strno] =
    my_realloc(g_store_inter_leftchar[strno], offset * sizeof(unsigned char));

#if STAT
  g_totalmem += (offset * sizeof(struct internode));
  g_totalmem += (offset * sizeof(NODEID));
#endif /* STAT */

  p = g_store_inter[strno];
  p_s = g_store_inter_suffixlink[strno];
  p_l = g_store_inter_leftchar[strno];
  i = m_NODEID_get_MEMUNIT_N(smuaP->memunit[smuaP->unit_current].str_base_NODEID);
  for ( ; (smuaP->memunit[i].strno == strno) && (i<= smuaP->unit_current)
	 ; i++) {
    smuaP->memunit[i].orig.inter = p;
    smuaP->memunit[i].orig_suffixlink = p_s;
    smuaP->memunit[i].orig_leftchar = p_l;
    p += G_NODEID_INTER_MEMUNIT_OFFSETS;
    p_s += G_NODEID_INTER_MEMUNIT_OFFSETS;
    p_l += G_NODEID_INTER_MEMUNIT_OFFSETS;
  }
  return;
} /* end of func */

int stb_get_leafno(NODEID leaf)
/* leafno begins with 0 */
{
  NODEID unit = m_NODEID_get_MEMUNIT_N(leaf);

  return leaf - g_memunit_admin_leaf.memunit[unit].str_base_NODEID;
} /* end of func */

void stmem_free_suffixlink(void)
{
  STRNO_T i;

  for (i = 0; i < g_num_of_strings; i++) {
    my_free(g_store_inter_suffixlink[i]);
    g_store_inter_suffixlink[i] = NULL;
  } /* for */

#if STAT
  /* this may be incorrect */
  g_totalmem -= (g_count_internode * sizeof(NODEID));
#endif

  return;
} /* end of func */

#if 0
NODEID stmem_new_internode(void)
{
  struct st_MEMUNIT_admin *smuaP = &g_memunit_admin_inter;
  NODEID ret_id;
  NODEID id_base;
  NODEID cur;
  NODEID offs;
  STRNO_T strno;

  ret_id = smuaP->id_current;
  offs = smuaP->offset_current;
  cur = smuaP->unit_current;

  smuaP->memunit[cur].orig.inter[offs].s = 0;
  smuaP->memunit[cur].orig.inter[offs].neighbor = NODEID_NULL;
  smuaP->memunit[cur].orig.inter[offs].len = 0;
  smuaP->memunit[cur].orig.inter[offs].child = NODEID_NULL;
  smuaP->memunit[cur].orig.inter[offs].strno = 0;
  smuaP->memunit[cur].orig_suffixlink[offs] = NODEID_NULL;

  smuaP->id_current += 1;
  smuaP->offset_current += 1;

  if (smuaP->offset_current >= G_NODEID_INTER_MEMUNIT_OFFSETS) {
    smuaP->offset_current = 0;
    id_base = smuaP->memunit[cur].str_base_NODEID;
    strno = smuaP->memunit[cur].strno;
    st_newalloc_MEMUNIT_inter(&(smuaP->memunit[cur + 1]), strno,
			      id_base);
    smuaP->unit_current += 1;
  }
#if STAT
  g_count_internode += 1;
#endif
  return ret_id;
} /* end of func */
#endif /* 0 */

NODEID stmem_new_internode(void)
{
  struct st_MEMUNIT_admin *smuaP = &g_memunit_admin_inter;
  NODEID ret_id;
  NODEID id_base;
  NODEID cur;
  NODEID offs;
  STRNO_T strno;


  ret_id = smuaP->id_current;
  offs = smuaP->offset_current;
  cur = smuaP->unit_current;

  smuaP->memunit[cur].orig.inter[offs].s = 0;
  smuaP->memunit[cur].orig.inter[offs].neighbor = NODEID_NULL;
  smuaP->memunit[cur].orig.inter[offs].len = 0;
  smuaP->memunit[cur].orig.inter[offs].child = NODEID_NULL;
  smuaP->memunit[cur].orig_suffixlink[offs] = NODEID_NULL;

  smuaP->id_current += 1;
  smuaP->offset_current += 1;

  if (smuaP->offset_current >= G_NODEID_MEMUNIT_OFFSETS) {
    smuaP->offset_current = 0;
    id_base = smuaP->memunit[cur].str_base_NODEID;
    strno = smuaP->memunit[cur].strno;
    smuaP->unit_current += 1;
    cur += 1;
    g_store_inter_offset += G_NODEID_MEMUNIT_OFFSETS;
    smuaP->memunit[cur].orig.inter =
      g_store_inter[strno] + g_store_inter_offset;
    smuaP->memunit[cur].orig_suffixlink = 
      g_store_inter_suffixlink[strno] + g_store_inter_offset;
    smuaP->memunit[cur].orig_leftchar = 
      g_store_inter_leftchar[strno] + g_store_inter_offset;
    st_MEMUNIT_set_strno(&(smuaP->memunit[cur]), strno,
			 id_base);
  }
#if STAT
  g_count_internode += 1;
#endif /* STAT */
  return ret_id;
} /* end of func */

NODEID stmem_new_leaf(void)
{
  struct st_MEMUNIT_admin *smuaP = &g_memunit_admin_leaf;
  NODEID ret_id;
  NODEID id_base;
  NODEID cur;
  NODEID offs;
  STRNO_T strno;

  ret_id = smuaP->id_current;
  offs = smuaP->offset_current;
  cur = smuaP->unit_current;

  smuaP->memunit[cur].orig.leaf[offs].s = 0;
  smuaP->memunit[cur].orig.leaf[offs].neighbor = NODEID_NULL;
  smuaP->memunit[cur].orig_leftchar[offs] = g_nextleftchar;
  g_nextleftchar = *g_currentleftchar++;

  smuaP->id_current += 1;
  smuaP->offset_current += 1;
  if (smuaP->offset_current >= G_NODEID_MEMUNIT_OFFSETS) {
    smuaP->offset_current = 0;
    id_base = smuaP->memunit[cur].str_base_NODEID;
    strno = smuaP->memunit[cur].strno;
    smuaP->unit_current += 1;
    cur += 1;
    g_store_leaf_offset += G_NODEID_MEMUNIT_OFFSETS;
    smuaP->memunit[cur].orig.leaf = g_store_leaf[strno] + g_store_leaf_offset;
    smuaP->memunit[cur].orig_leftchar = 
      g_store_leaf_leftchar[strno] + g_store_leaf_offset;
    st_MEMUNIT_set_strno(&(smuaP->memunit[cur]), strno,
			 id_base);
  }
#if STAT
  g_count_leaf += 1;
#endif /* STAT */
  return ret_id;
} /* end of func */

/* ***************************************************************** */

static int m_stb_node_getlen(NODEID id)
{
  struct st_MEMUNIT_admin *smuaP;
  NODEID unit;
  NODEID offset;

  if (m_stb_self_is(id)) {
    /* self is LEAF */
    unit = m_NODEID_get_MEMUNIT_N(id);
    offset = m_NODEID_get_MEMUNIT_OFFSET(id);
    smuaP = &g_memunit_admin_leaf;
    return g_strlen[smuaP->memunit[unit].strno] - 
      smuaP->memunit[unit].orig.leaf[offset].s;
  } else {
    /* self is INTER */
    unit = m_NODEID_INTER_get_MEMUNIT_N(id);
    offset = m_NODEID_INTER_get_MEMUNIT_OFFSET(id);
    smuaP = &g_memunit_admin_inter;
    return smuaP->memunit[unit].orig.inter[offset].len;
  }
} /* end of func */

static int m_stb_node_getlen_leaf(NODEID id)
{
  struct st_MEMUNIT_admin *smuaP;
  NODEID unit;
  NODEID offset;

  /* self is LEAF */
  unit = m_NODEID_get_MEMUNIT_N(id);
  offset = m_NODEID_get_MEMUNIT_OFFSET(id);
  smuaP = &g_memunit_admin_leaf;
  return g_strlen[smuaP->memunit[unit].strno] - 
    smuaP->memunit[unit].orig.inter[offset].s;
} /* end of func */

static int m_stb_node_getlen_inter(NODEID id)
{
  struct st_MEMUNIT_admin *smuaP;
  NODEID unit;
  NODEID offset;

  /* self is INTER */
  unit = m_NODEID_INTER_get_MEMUNIT_N(id);
  offset = m_NODEID_INTER_get_MEMUNIT_OFFSET(id);
  smuaP = &g_memunit_admin_inter;
  return smuaP->memunit[unit].orig.inter[offset].len;
} /* end of func */


static NODEID m_NODE_getmember_suffixlink(NODEID id)
{
  struct st_MEMUNIT_admin *smuaP;
  NODEID unit;
  NODEID offset;

  /* only for INTER */
  smuaP = &g_memunit_admin_inter;
  unit = m_NODEID_INTER_get_MEMUNIT_N(id);
  offset = m_NODEID_INTER_get_MEMUNIT_OFFSET(id);

  return smuaP->memunit[unit].orig_suffixlink[offset];
} /* end of func */

static void m_NODE_setmember_suffixlink(NODEID id, NODEID val_suffixlink)
{
  struct st_MEMUNIT_admin *smuaP;
  NODEID unit;
  NODEID offset;

  /* only for INTER */
  smuaP = &g_memunit_admin_inter;
  unit = m_NODEID_INTER_get_MEMUNIT_N(id);
  offset = m_NODEID_INTER_get_MEMUNIT_OFFSET(id);

  smuaP->memunit[unit].orig_suffixlink[offset] = val_suffixlink;
} /* end of func */

static unsigned char m_NODE_getmember_leftchar(NODEID id)
{
  struct st_MEMUNIT_admin *smuaP;
  NODEID unit;
  NODEID offset;

  if (m_stb_self_is(id)) {
    /* self is LEAF */
    smuaP = &g_memunit_admin_leaf;
    unit = m_NODEID_get_MEMUNIT_N(id);
    offset = m_NODEID_get_MEMUNIT_OFFSET(id);
  } else {
    /* self is INTER */
    smuaP = &g_memunit_admin_inter;
    unit = m_NODEID_INTER_get_MEMUNIT_N(id);
    offset = m_NODEID_INTER_get_MEMUNIT_OFFSET(id);
  }

  return smuaP->memunit[unit].orig_leftchar[offset];
} /* end of func */

static void m_NODE_setmember_leftchar(NODEID id, unsigned char val_lc)
{
  struct st_MEMUNIT_admin *smuaP;
  NODEID unit;
  NODEID offset;

  if (m_stb_self_is(id)) {
    /* self is LEAF */
    smuaP = &g_memunit_admin_leaf;
    unit = m_NODEID_get_MEMUNIT_N(id);
    offset = m_NODEID_get_MEMUNIT_OFFSET(id);
  } else {
    /* self is INTER */
    smuaP = &g_memunit_admin_inter;
    unit = m_NODEID_INTER_get_MEMUNIT_N(id);
    offset = m_NODEID_INTER_get_MEMUNIT_OFFSET(id);
  }

  smuaP->memunit[unit].orig_leftchar[offset] = val_lc;
} /* end of func */


static INTER *m_NODEID2ptr_inter(NODEID id)
{
  NODEID unit = m_NODEID_INTER_get_MEMUNIT_N(id);
  NODEID offset = m_NODEID_INTER_get_MEMUNIT_OFFSET(id);

  return &(g_memunit_admin_inter.memunit[unit].orig.inter[offset]);

} /* end of func */

static LEAF *m_NODEID2ptr_leaf(NODEID id)
{
  NODEID unit = m_NODEID_get_MEMUNIT_N(id);
  NODEID offset = m_NODEID_get_MEMUNIT_OFFSET(id);

  return &(g_memunit_admin_leaf.memunit[unit].orig.leaf[offset]);

} /* end of func */

static STRNO_T m_NODE_getmember_strno(NODEID id)
{
  struct st_MEMUNIT_admin *smuaP;
  NODEID unit;

  if (m_stb_self_is(id)) {
    /* self is LEAF */
    smuaP = &g_memunit_admin_leaf;
    unit = m_NODEID_get_MEMUNIT_N(id);
    return smuaP->memunit[unit].strno;
  } else {
    /* self is INTER */
    smuaP = &g_memunit_admin_inter;
    unit = m_NODEID_get_MEMUNIT_N(id);
    return smuaP->memunit[unit].strno;
  }
} /* end of func */

#if 0
static void m_NODE_setmember_strno(NODEID id, STRNO_T val_strno)
{
  struct st_MEMUNIT_admin *smuaP;
  NODEPTR ptr;

  if (m_stb_self_is(id)) {
    /* self is LEAF */
    return;
  } else {
    /* self is INTER */
    return;
  }
} /* end of func */
#endif /* 0 */

static int m_stb_self_is(NODEID n)
{
  return (n >> G_NODEID_MEMUNIT_FLAGBIT);
} /* end of func */

/* wrappers */

static int m_NODE_getmember_s(NODEID id)
{
  NODEPTR ptr;

  if (m_stb_self_is(id)) {
    /* self is LEAF */
    ptr.leaf = m_NODEID2ptr_leaf(id);
    return ptr.leaf->s;
  } else {
    /* self is INTER */
    ptr.inter = m_NODEID2ptr_inter(id);
    return ptr.inter->s;
  }
} /* end of func */

static void m_NODE_setmember_s(NODEID id, int val_s)
{
  NODEPTR ptr;

  if (m_stb_self_is(id)) {
    /* self is LEAF */
    ptr.leaf = m_NODEID2ptr_leaf(id);
    ptr.leaf->s = val_s;
  } else {
    /* self is INTER */
    ptr.inter = m_NODEID2ptr_inter(id);
    ptr.inter->s = val_s;
  }
  return;
} /* end of func */

static void m_NODE_add_member_s(NODEID id, int val)
{
  NODEPTR ptr;

  if (m_stb_self_is(id)) {
    /* self is LEAF */
    ptr.leaf = m_NODEID2ptr_leaf(id);
    ptr.leaf->s += val;
  } else {
    /* self is INTER */
    ptr.inter = m_NODEID2ptr_inter(id);
    ptr.inter->s += val;
  }
  return;
} /* end of func */

static NODEID m_NODE_getmember_neighbor(NODEID id)
{
  NODEPTR ptr;

  if (m_stb_self_is(id)) {
    /* self is LEAF */
    ptr.leaf = m_NODEID2ptr_leaf(id);
    return ptr.leaf->neighbor;
  } else {
    /* self is INTER */
    ptr.inter = m_NODEID2ptr_inter(id);
    return ptr.inter->neighbor;
  }
} /* end of func */

static void m_NODE_setmember_neighbor(NODEID id, NODEID val_neighbor)
{
  NODEPTR ptr;

  if (m_stb_self_is(id)) {
    /* self is LEAF */
    ptr.leaf = m_NODEID2ptr_leaf(id);
    ptr.leaf->neighbor = val_neighbor;
  } else {
    /* self is INTER */
    ptr.inter = m_NODEID2ptr_inter(id);
    ptr.inter->neighbor = val_neighbor;
  }
  return;
} /* end of func */

static int m_NODE_getmember_len(NODEID id)
{
  NODEPTR ptr;

  /* only for INTER */
  ptr.inter = m_NODEID2ptr_inter(id);
  return ptr.inter->len;
} /* end of func */

static void m_NODE_setmember_len(NODEID id, int val_len)
{
  NODEPTR ptr;

  /* only for INTER */
  ptr.inter = m_NODEID2ptr_inter(id);
  ptr.inter->len = val_len;
  return;
} /* end of func */

static void m_NODE_sub_member_len(NODEID id, int val)
{
  NODEPTR ptr;

  /* only for INTER */
  ptr.inter = m_NODEID2ptr_inter(id);
  ptr.inter->len -= val;
  return;
} /* end of func */

static NODEID m_NODE_getmember_child(NODEID id)
{
  NODEPTR ptr;

  /* only for INTER */
  ptr.inter = m_NODEID2ptr_inter(id);
  return ptr.inter->child;
} /* end of func */

static void m_NODE_setmember_child(NODEID id, NODEID val_child)
{
  NODEPTR ptr;

  /* only for INTER */
  ptr.inter = m_NODEID2ptr_inter(id);
  ptr.inter->child = val_child;
  return;
} /* end of func */


