/*
  my_malloc_debug.c -- malloc wrapper for debugging

    Copyright (C) 1999,2000 Naohisa Goto <ngoto@gen-info.osaka-u.ac.jp>
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* #define _REENTRANT */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <unistd.h>
#include "my_malloc.h"

/* MY_MALLOC_DEBUG_LIST: malloc history list size */
/* ex: gcc -DMY_MALLOC_DEBUG_LIST=50000 xxx.c my_malloc_debug.c */

#ifndef MY_MALLOC_DEBUG_LIST
#define MY_MALLOC_DEBUG_LIST 10000
#endif /* MY_MALLOC_DEBUG_LIST */

static struct {
  void *ptr;
  size_t size;
} allocedarea[MY_MALLOC_DEBUG_LIST];

static int current = 0;

void malloced_but_not_free_area(void)
{
  int i;
  fprintf(stderr, "atexit...\n");
  for (i = 0; i < MY_MALLOC_DEBUG_LIST; i++) {
    if (allocedarea[i].ptr != NULL) {
      fprintf(stderr, "not free: %p %ld ==> free it now!\n", 
	      allocedarea[i].ptr, (long)allocedarea[i].size);
      free(allocedarea[i].ptr);
    }
  } /* for */
} /* end of func */

void add_allocedarea(void *p, long size)
{
  int i;
  static int unregistered = 1;
  if (unregistered) {
    atexit(malloced_but_not_free_area);
    unregistered = 0;
  }

  allocedarea[current].ptr = p;
  allocedarea[current].size = size;

  i = 0;
  for (i = 1; i < MY_MALLOC_DEBUG_LIST; i++) {
    if (allocedarea[(current + i) % MY_MALLOC_DEBUG_LIST].ptr == NULL) {
      current = (current + i) % MY_MALLOC_DEBUG_LIST;
      return;
    }
  } /* for */
  fprintf(stderr, "my_malloc_debug.c: alloc table for debug overflow\n");
  current = (current + 1) % MY_MALLOC_DEBUG_LIST;
  return;
} /* end of func */

long del_allocedarea(void *p)
{
  int i;
  if (p == NULL) return -1;
  for (i = 0; i < MY_MALLOC_DEBUG_LIST; i++) {
    if (allocedarea[i].ptr == p) {
      allocedarea[i].ptr = NULL;
      return (long)(allocedarea[i].size);
    }
  } /* for */
  return -9999;
} /* end of func */

void my_free(void *ptr)
{
  long debug_size;
  debug_size = del_allocedarea(ptr);
  fprintf(stderr, "free: %p %ld\n", ptr, debug_size); /* for DEBUG */
  free(ptr);
} /* end of func */

void *my_malloc(size_t size)
{
  void *p;

  p = malloc(size);
  if (p == NULL) {
    fprintf(stderr, "Memory allocate error; fatal, abort\n");
    exit(2);
  }

  add_allocedarea(p, size);
  fprintf(stderr, "malloc: %p %ld\n", p, (long)size); /* for DEBUG */
  return p;
} /* end of func */

void *my_calloc(size_t nelem, size_t elsize)
{
  void *p;

  p = calloc(nelem, elsize);
  if (p == NULL) {
    fprintf(stderr, "Memory allocate error; fatal, abort\n");
    exit(2);
  }

  add_allocedarea(p, nelem * elsize);
  fprintf(stderr, "calloc: %p %ld\n", p, (long)nelem * elsize); /* for DEBUG */
  return p;
} /* end of func */

void *my_realloc(void *ptr, size_t size)
{
  long oldsize;

  oldsize = del_allocedarea(ptr);
  fprintf(stderr, "realloc: %p %ld -> ", ptr, oldsize);

  ptr = realloc(ptr, size);
  fprintf(stderr, "%p %ld\n", ptr, (long)size);
  if (size != 0) {
    if (ptr == NULL) {
      fprintf(stderr, "realloc failed; fatal, abort\n");
      exit(2);
    }
    add_allocedarea(ptr, size);
  } else { /* size == 0 */
    if (ptr != NULL) {
      fprintf(stderr, "realloc failed; fatal, abort\n");
      exit(2);
    }
  }

  return ptr;
} /* end of func */

