/*
  ifilter.c -- input filter functions for CONSERV

    Copyright (C) 1999,2000 Naohisa Goto <ngoto@gen-info.osaka-u.ac.jp>
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <unistd.h>
#include <stddef.h>
#include "mfstlib3.h"
#include "my_malloc.h"
#include "ifilter.h"
#include "xmask_ifilter.h"

static struct ifilter_activelist *ifilter_activelist_rewind(
 struct ifilter_activelist *alist);
static struct ifilter_activelist *ifilter_activelist_move_to_last(
 struct ifilter_activelist *alist);
static struct ifilter_activelist *ifilter_activelist_grep(
 struct ifilter_activelist *alist, struct ifilter_namelist *namelist);
static void ifilter_close_single(struct ifilter_activelist *alist);
static struct ifilter_activelist *ifilter_open_single(
 struct ifilter_namelist *namelist, uchar *optstr);

void ifilter_print_help(struct ifilter_namelist *nlist, FILE *fpo)
{
  fprintf(fpo, "\n");
  fprintf(fpo, "Available input filters:\n");
  while (nlist != NULL) {
    nlist->details->help(nlist->name, fpo);
    nlist = nlist->next;
  }
  return;
} /* end of func */

void ifilter_print_activelist(struct ifilter_activelist *alist,
 FILE *fpo, uchar *prefix)
{
  alist = ifilter_activelist_rewind(alist);
  while (alist != NULL) {
    fprintf(fpo, "%sinput_filter: %s=%s\n",
	    prefix, alist->namelist->name, alist->optstr);
    alist = alist->next;
  } /* while */
  return;
} /* end of func */

void ifilter_result_free(struct ifilter_result *ifr)
{
  struct ifilter_result *ifr_next;

  while (ifr != NULL) {
    ifr_next = ifr->next;
    my_free(ifr);
    ifr = ifr_next;
  }
  return;
} /* end of func */

struct ifilter_result *ifilter_main(uchar *str_in, int len_in,
 struct ifilter_activelist *alist)
{
  int i;
  int nlen, noffset;
  uchar *nstr;
  uchar *str;
  struct ifilter_result *rnew;
  struct ifilter_result **rcurrent;
  struct ifilter_result *rtop = NULL;
  rcurrent = &rtop;

  alist = ifilter_activelist_rewind(alist);

  while (alist != NULL) {
    alist->namelist->details->main(str_in, len_in, alist->settings);
    alist = alist->next;
  }

  nlen = 0;
  noffset = 0;
  str = str_in;
  nstr = str;
  for (i = 0; i <= len_in; i++) { /* include last '\0' in this loop */
    if (*str++ == '\0') {
      if (nlen > 0) {
	rnew = my_calloc(1, sizeof(struct ifilter_result));
	rnew->str = nstr;
	rnew->len = nlen;
	rnew->offset = noffset;
	rnew->offset_c = len_in - noffset;
	rnew->next = NULL;
	*rcurrent = rnew;
	rcurrent = &(rnew->next);
      }
      nlen = 0;
      nstr = str; /* already +1 in if(*str++...) */
      noffset = i + 1;
    } else {
      nlen += 1;
    }
  } /* for */

  return rtop;
} /* end of func */

struct ifilter_activelist *ifilter_registration(uchar *optarg,
 struct ifilter_namelist *namelist, struct ifilter_activelist *alist)
{
  uchar *optstr;
  int namelen;
  struct ifilter_activelist *alist_new, *alist_g;

  while (namelist != NULL) {
    namelen = strlen(namelist->name);
    if (strncmp(optarg, namelist->name, namelen) == 0) {
      optstr = optarg + namelen;
      switch (optarg[namelen]) {
      case '=':
	optstr += 1;
	/* fall into next case */
      case '\0':
	alist_new = ifilter_open_single(namelist, optstr);
	if (alist_new == NULL) {
	  fprintf(stderr, "input filter \"%s\" open error\n", namelist->name);
	  return alist;
	}
	if (namelist->attribute == IFILTER_OVERRIDE) {
	  alist_g = ifilter_activelist_grep(alist, namelist);
	  if (alist != NULL) {
	    alist_new->next = alist_g->next;
	    alist_new->before = alist_g->before;
	    ifilter_close_single(alist_g);
	    if (alist_new->next != NULL) alist_new->next->before = alist_new;
	    if (alist_new->before != NULL) alist_new->before->next = alist_new;
	    fprintf(stderr, "overriding input filter: %s %s\n",
		    namelist->name, optstr);
	    return alist_new;
	  } else {
	    alist = alist_g;
	  }
	} /* if (namelist->attribulte) */
	alist = ifilter_activelist_move_to_last(alist);
	alist_new->before = alist;
	if (alist != NULL) alist->next = alist_new;
	fprintf(stderr, "using input filter: %s %s\n",
		namelist->name, optstr);
	return alist_new;
	break;
      default:
	break;
      } /* switch */
    } /* if */
    namelist = namelist->next;
  } /* while */

  fprintf(stderr, "no input filter named \"%s\"\n", optarg);
  return alist;
} /* end of func */

static struct ifilter_activelist *ifilter_activelist_rewind(
 struct ifilter_activelist *alist)
{
  struct ifilter_activelist *p;

  p = alist;
  while (p != NULL) {
    alist = p;
    p = p->before;
  }
  return alist;
} /* end of func */

static struct ifilter_activelist *ifilter_activelist_move_to_last(
 struct ifilter_activelist *alist)
{
  struct ifilter_activelist *p;

  p = alist;
  while (p != NULL) {
    alist = p;
    p = p->next;
  }
  return alist;
} /* end of func */

static struct ifilter_activelist *ifilter_activelist_grep(
 struct ifilter_activelist *alist, struct ifilter_namelist *namelist)
{
  alist = ifilter_activelist_rewind(alist);
  while (alist != NULL) {
    if (alist->namelist == namelist) break;
    alist = alist->next;
  }
  return alist;
} /* end of func */

void ifilter_close_all(struct ifilter_activelist *alist)
{
  struct ifilter_activelist *p, *q;

  p = ifilter_activelist_rewind(alist);
  while (p != NULL) {
    q = p->next;
    ifilter_close_single(p);
    p = q;
  }

  return;
} /* end of func */

static void ifilter_close_single(struct ifilter_activelist *alist)
{
  alist->namelist->details->close(alist->settings);
  my_free(alist->optstr);
  my_free(alist);
  return;
} /* end of func */

static struct ifilter_activelist *ifilter_open_single(
 struct ifilter_namelist *namelist, uchar *optstr)
{
  struct ifilter_activelist *alist;
  uchar *optstr_new;
  void *settings;

  settings = namelist->details->open(optstr, &optstr_new);
  if (settings == NULL) return NULL;

  alist = my_malloc(sizeof(struct ifilter_activelist));
  alist->next = NULL;
  alist->before = NULL;
  alist->namelist = namelist;
  alist->settings = settings;
  alist->optstr = optstr_new;

  return alist;
} /* end of func */
