/*
  xmask_ifilter.c -- input filter: masking X or N

    Copyright (C) 1999-2005 Naohisa Goto <ngoto@gen-info.osaka-u.ac.jp>
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* xmask_ifilter.c */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>
#include <unistd.h>
#include <stddef.h>
#include "my_malloc.h"
#include "ifilter.h"
#include "xmask_ifilter.h"

/* string length NOT include last '\0' */
static void xmask_ifilter_help(const uchar *name, FILE *fpo);
static void *xmask_ifilter_open(uchar *optarg, uchar **optstr_new);
static void xmask_ifilter_close(void *settings);
static int xmask_ifilter_main(uchar *str_in, int len_in, void *settings);

struct xmask_ifilter_settings {
  uchar *maskchar;
  int masklen_min;
};

/* global variables */
const struct ifilter_details xmask_ifilter_details = {
  xmask_ifilter_open,
  xmask_ifilter_close,
  xmask_ifilter_main,
  xmask_ifilter_help
};

static void xmask_ifilter_help(const uchar *name, FILE *fpo)
{
  fprintf(fpo, "    xmask: mask repeats of specified letters\n");
  fprintf(fpo, "        xmask[=[length][,letters]]\n"
	       "            length      minimum length of repeat\n"
	       "            letters     letters to be masked\n"
	       "        default options: 10,XN (xmask=10,XN)\n"
	);
  return;
} /* end of func */

static void *xmask_ifilter_open(uchar *optarg, uchar **optstr_new)
{
  const uchar *mstr = "XN";
  int mlen = 10;
  struct xmask_ifilter_settings *xset;
  uchar *s;
  int slen;
  int tmp1 = 0, tmp2 = 0;
  int i, r;

  if (optarg[0] != '\0' && optarg[0] != ',') {
    r = sscanf(optarg, "%d%n", &tmp1, &tmp2);
    if (r > 0) {
      optarg += tmp2;
      if (tmp1 > 0) {
	mlen = tmp1;
      } else {
	fprintf(stderr, "xmask: option error: out of range\n");
	return NULL;
      }
    }
  } /* if */

  if (optarg[0] == ',') {
    optarg++;
    mstr = optarg;
  }

  xset = my_malloc(sizeof(struct xmask_ifilter_settings));
  slen = strlen(mstr);
  s = my_malloc(slen + 1);
  strncpy(s, mstr, slen);
  s[slen] = '\0';

  xset->maskchar = s;
  xset->masklen_min = mlen;

  for (i = mlen; i > 0; i /= 10) slen += 1;
  slen += 3;
  s = my_malloc(slen);
  snprintf(s, slen, "%d,%s", mlen, mstr);
  *optstr_new = s;

  return (void *)xset;
} /* end of func */

static void xmask_ifilter_close(void *settings)
{
  struct xmask_ifilter_settings *xset = settings;

  my_free(xset->maskchar);
  my_free(xset);
  return;
} /* end of func */

static int xmask_ifilter_main(uchar *str_in, int len_in, void *settings)
{
  struct xmask_ifilter_settings *xset;
  uchar *xstr;
  uchar *mask;
  int i, j;
  int c, d;
  int mode = 0;
  uchar *str = str_in;
  int xlen;
  int xflag = 0;

  xset = (struct xmask_ifilter_settings *)settings;
  xstr = NULL;
  xlen = 0;

  for (i = 0; i <= len_in; i++) { /* including last '\0' */
    c = *str;
    mask = xset->maskchar;
    while ((d = *mask++) != '\0') {
      if (c == d) break;
    }

    switch (mode) {
    case 0: /* normal mode */
      if (d != '\0') { /* change to in-mask-string mode */
	mode = 1; /* in mask string */
	xstr = str;
	xlen = 1;
	xflag = 0;
      }
      break;
    case 1: /* in mask string */
      if (xlen >= xset->masklen_min) xflag = 1;
      if (d != '\0') { /* mask char */
	xlen += 1;
      } else {
	mode = 0; /* change to normal mode */
	if (xflag) {
	  for (j = 0; j < xlen; j++) *xstr++ = '\0';
	} /* if (xflag) */
      } /* if (d != '\0') */
      break;
    } /* switch */
    str++;
  } /* for */

  return 0;
} /* end of func */

